"""Base class for XMS DMI coverage components."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import csv
from enum import IntEnum
import os

# 2. Third party modules
import pandas as pd

# 3. Aquaveo modules
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.components.bases.component_base import ActionRv, ComponentBase
from xms.components.display.display_options_io import read_display_option_ids


class ColAttType(IntEnum):
    """An enumeration of possible shapefile field data types."""
    COL_ATT_INT = 0
    COL_ATT_DBL = 1
    COL_ATT_STR = 2


class CoverageComponentBase(ComponentBase):
    """Base class for XMS DMI coverage components."""
    def __init__(self, main_file):
        """Construct the base class.

        Args:
            main_file (str): Every component should take a main file path as an argument at construction. This
                file will be used by XMS to instantiate Python instances of components. Any persistent data of
                the component should be stored in this file.
        """
        super().__init__(main_file)
        self.comp_to_xms = {}  # {cov_uuid: {entity_type: {comp_id: [att_id]}}} - Contents reflect current state in XMS
        self.update_ids = {}  # {cov_uuid: {entity_type: {att_id: comp_id}}} - Contents sent to XMS after ActionRequest
        self.label_texts = {}  # {cov_uuid: {entity_type: {comp_id: label_text (str)}}} - Labels for individual features
        self.uuid = ''  # Set this before adding any id mappings.
        self.cov_uuid = ''  # Set this before adding any id mappings. Change to switch coverage being mapped.

    @staticmethod
    def get_entity_type_from_string(entity_type):
        """Convert entity type string to a TargetType enum value.

        Args:
            entity_type (str): One of 'POINT', 'ARC', or 'POLYGON'

        Returns:
            TargetType: See description
        """
        if entity_type == 'POINT':
            return int(TargetType.point)
        elif entity_type == 'ARC':
            return int(TargetType.arc)
        elif entity_type == 'ARC_GROUP':
            return int(TargetType.arc_group)
        elif entity_type == 'POLYGON':
            return int(TargetType.polygon)

    @staticmethod
    def write_table_def(filename, columns):
        """Write a coverage shapefile column definition file.

        Args:
            filename (str): Path to the file to write
            columns (list): List of tuples where first element of each tuple is the column name and second element is
                ColAttType
        """
        with open(filename, 'w') as f:
            f.write(f'NROWCOL 0 {len(columns)}\n')
            for column in columns:
                f.write('BEGCOL\n')
                f.write(f'COLATT \"{column[0]}\" {int(column[1])} 0 NONE\n\n')
                f.write('XYSERIES 0\nENDCOL\n')
            f.write('ENDTABLE')

    def _write_entity_att_table(self, base_name, target_type, write_atts):
        """Write shapefile atts for a feature object type to a csv file with a table definition file.

        Args:
            base_name (str): Base filepath for definition and table att files
            target_type (TargetType): The feature object type
            write_atts (bool): If False, only the table definition file will be written.

        Returns:
            (str, str): The definition filepath, the table att csv file
        """
        def_file = f'{base_name}.def'
        att_file = f'{base_name}.table'
        cols = self.get_table_def(target_type)
        if cols:
            self.write_table_def(def_file, cols)
            if write_atts:
                df = self.get_att_table(target_type)
                if df is not None:
                    df.to_csv(att_file, index=False, quoting=csv.QUOTE_NONNUMERIC)
        return def_file, att_file

    def get_xms_ids(self, entity_type, comp_id):
        """Find the XMS ids on the current coverage associated with a component id for an entity type.

        Args:
            entity_type (xms.guipy.data.TargetType): The type of entity this mapping is for. Valid values are
                TargetType.point, TargetType.arc, TargetType.polygon.
            comp_id (int): The component id to look up.

        Returns:
            (:obj:`list` of :obj:`int`): The current XMS ids associated with a component id.
        """
        try:
            return self.comp_to_xms[self.cov_uuid][entity_type][comp_id]
        except KeyError:
            return -1

    def get_comp_id(self, entity_type, xms_id):
        """Find the component id on the current coverage currently associated with an XMS id for an entity type.

        Args:
            entity_type (xms.guipy.data.TargetType): The type of entity this mapping is for. Valid values are
                TargetType.point, TargetType.arc, TargetType.polygon.
            xms_id (int): The XMS id to look up.

        Returns:
            (int): The component id currently associated with xms_id
        """
        try:
            for comp_id, xms_ids in self.comp_to_xms[self.cov_uuid][entity_type].items():
                for att_id in xms_ids:
                    if att_id == xms_id:
                        return comp_id
        except KeyError:
            return -1

    def set_label_text(self, target_type, comp_id, label_text):
        """Set the specified display label text for a component id.

        Args:
            target_type (TargetType): The entity type of the feature
            comp_id (int): The component id of the feature
            label_text (str): Display label text for the feature
        """
        entity_dict = self.label_texts.setdefault(self.cov_uuid, {}).setdefault(target_type, {})
        entity_dict[comp_id] = label_text

    def get_label_texts(self, target_type, comp_ids):
        """Get the specified display label texts for a set of component ids.

        Args:
            target_type (TargetType): The entity type of the features
            comp_ids (list[int]): The component ids of the features

        Returns:
            list[str]: The display text for the component ids, empty string if no display text defined for the
                component id. Parallel with input `comp_ids`.
        """
        entity_dict = self.label_texts.get(self.cov_uuid, {}).get(target_type, {})
        label_texts = [entity_dict.get(comp_id, '') for comp_id in comp_ids]
        if not any(label_texts):
            return None  # Return None if no label texts have been defined.
        return label_texts

    def update_component_id(self, entity_type, xms_id, comp_id, label_text=None):
        """Update a coverage component id mapping to be sent to XMS after an ActionRequest.

        Args:
            entity_type (xms.guipy.data.TargetType): The type of entity this mapping is for. Valid values are
                TargetType.point, TargetType.arc, TargetType.polygon.
            xms_id (int): The entity's current id in XMS. Should get from XMS to ensure it is up to date.
            comp_id (int): The component id to associate with this entity.
            label_text (Optional[str]): Display label text for this component id. Will be the display category
                description text if not specified.
        """
        if self.cov_uuid not in self.update_ids:
            self.update_ids[self.cov_uuid] = {}
            self.label_texts[self.cov_uuid] = {}
        int_type = int(entity_type)
        if int_type not in self.update_ids[self.cov_uuid]:
            self.update_ids[self.cov_uuid][int_type] = {}
            self.label_texts[self.cov_uuid][int_type] = {}
        self.update_ids[self.cov_uuid][int_type][xms_id] = comp_id
        if label_text is not None:
            self.label_texts[self.cov_uuid][int_type][comp_id] = label_text

    def load_coverage_component_id_map(self, file_map: dict[str, tuple[str, str]]) -> None:
        """Reads the binary id files dumped by XMS and populates the internal id map.

        Args:
            file_map: Key is entity type string, value is tuple of two str where the first is file location of the
             xms id file and the second is the file location of the component id file.
        """
        for entity_type, id_files in file_map.items():
            if len(id_files) < 2:
                continue
            # Read the parallel binary id arrays
            xms_ids = read_display_option_ids(id_files[0])
            comp_ids = read_display_option_ids(id_files[1])
            if xms_ids:
                entity_type_enum = self.get_entity_type_from_string(entity_type)
                if self.cov_uuid not in self.comp_to_xms:
                    self.comp_to_xms[self.cov_uuid] = {}
                if entity_type_enum not in self.comp_to_xms[self.cov_uuid]:
                    self.comp_to_xms[self.cov_uuid][entity_type_enum] = {}
                for xms_id, comp_id in zip(xms_ids, comp_ids):
                    if comp_id not in self.comp_to_xms[self.cov_uuid][entity_type_enum]:
                        self.comp_to_xms[self.cov_uuid][entity_type_enum][comp_id] = []
                    self.comp_to_xms[self.cov_uuid][entity_type_enum][comp_id].append(xms_id)

    def get_component_coverage_ids(self):
        """Method used by coverage component implementations for getting id mappings.

        Returns:
            (tuple of str, dict): The component UUID and the coverage component id mapping.
        """
        return self.uuid, self.update_ids

    def handle_merge(self, merge_list: list[tuple[str, dict]]) -> ActionRv:
        """Method used by coverage component implementations to handle coverage merges.

        Default implementation for coverage components calls the create event.

        Args:
            merge_list: tuple containing 1) main_file: The absolute path to the main file of the old component this
             component is being merged from. 2) id_files: The dictionary keys are 'POINT', 'ARC', and 'POLYGON'. Each
             value is a tuple that may have two absolute file paths or none. The first file is for the ids in XMS on
             the coverage. The second file contains the ids the old component used for those objects. Both id files
             should be equal in length. This dictionary is only applicable if the component derives from
             CoverageComponentBase.

        Returns:
            (tuple): tuple containing:
                - messages: List of tuples with the first element of the tuple being the message level (DEBUG, ERROR,
                  WARNING, INFO) and the second element being the message text.
                - action_requests: List of actions for XMS to perform.
        """
        # No idea what the lock state is. Since this coming from a merge, it is whatever the XML defines.
        return self.create_event(lock_state=False)

    def write_shapefile_atts(self, query, params):
        """Implement on Coverage components to export attributes to a shapefile.

        Args:
            query (:obj:'xms.api.dmi.Query'): Query with a context at the component instance level.
            params: Contains 'target_type' enum list

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        if 'write_atts' in params[0] and params[0]['write_atts']:
            query.load_component_ids(self, points=True, arcs=True, polygons=True)
            write_atts = True
        else:
            write_atts = False  # Only write table definition files

        if 'point_base_file' in params[0]:
            base_name = params[0]['point_base_file']
            index = int(TargetType.point)
            self.att_files[index] = self._write_entity_att_table(base_name, TargetType.point, write_atts)

        if 'arc_base_file' in params[0]:
            base_name = params[0]['arc_base_file']
            self.att_files[int(TargetType.arc)] = self._write_entity_att_table(base_name, TargetType.arc, write_atts)

        if 'polygon_base_file' in params[0]:
            base_name = params[0]['polygon_base_file']
            index = int(TargetType.polygon)
            self.att_files[index] = self._write_entity_att_table(base_name, TargetType.polygon, write_atts)

        return [], []

    def read_shapefile_atts(self, query, params):
        """Implement on Coverage components to populate attributes from a shapefile.

        Args:
            query (:obj:'xms.api.dmi.Query'): Query with a context at the component instance level.
            params: Contains 'filename' list

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        att_dfs = {
            TargetType.point: None,
            TargetType.arc: None,
            TargetType.polygon: None,
        }

        # Unpack the att table files, which may or may not exist.
        try:
            point_file = params[0]['point_base_file']
            att_dfs[TargetType.point] = pd.read_csv(point_file)
            os.remove(point_file)
        except Exception:
            pass

        try:
            arc_file = params[0]['arc_base_file']
            att_dfs[TargetType.arc] = pd.read_csv(arc_file)
            os.remove(arc_file)
        except Exception:
            pass

        try:
            poly_file = params[0]['polygon_base_file']
            att_dfs[TargetType.polygon] = pd.read_csv(poly_file)
            os.remove(poly_file)
        except Exception:
            pass

        return self.populate_from_att_tables(att_dfs)

    def get_table_def(self, target_type):
        """Get the shapefile attribute table definition for a feature object.

        Args:
            target_type (TargetType): The feature object type

        Returns:
            list: list of tuples where first element is column name and second element is ColAttType
        """
        return None

    def get_att_table(self, target_type):
        """Get the shapefile attribute table for a feature object.

        Args:
            target_type (TargetType): The feature object type

        Returns:
            pandas.DataFrame: The shapefile attributes to write
        """
        return None

    def populate_from_att_tables(self, att_dfs):
        """Populate attributes from an attribute table written by XMS.

        Args:
            att_dfs (dict): Dictionary of attribute pandas.DataFrame. Key is TargetType enum, value is None if not
                applicable.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list` of :obj:`xms.api.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        return [], []
