"""Module for SimComponentBase."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['SimComponentBase', 'MessagesAndRequests']

# 1. Standard Python modules
from abc import ABC, abstractmethod
from pathlib import Path
from typing import Any, final, Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.guipy.dialogs.message_box import message_with_ok

# 4. Local modules
from xms.components.bases.component_with_menus_base import ComponentWithMenusBase, MessagesAndRequests


class SimComponentBase(ComponentWithMenusBase, ABC):
    """
    A basic simulation component.

    A derived class can override `_open_model_control()` to implement a model control dialog, and `item_linked()` to
    handle when an item was linked to the simulation. It can also add more right-click commands to the simulation by
    appending to `self.tree_commands`.
    """
    def __init__(self, main_file: Optional[str | Path] = None):
        """
        Initialize the component.

        Args:
            main_file: Path to the component's main-file.
        """
        super().__init__(main_file)
        self.tree_commands.append(('Model Control...', self._internal_open_model_control))

    def _internal_open_model_control(self, query: Query, _params: list[dict], parent: QWidget) -> MessagesAndRequests:
        """
        Run the model control dialog.

        Args:
            query: Interprocess communication object.
            _params: Ignored.
            parent: Parent widget.

        Returns:
            Messages and requests.
        """
        dialog_name = f'{self.module_name}.model_control'
        self._open_model_control(dialog_name, query, parent)
        return self.messages, self.requests

    @abstractmethod
    def _open_model_control(self, dialog_name: str, query: Query, parent: QWidget):
        """
        Run the model control dialog.

        Args:
            dialog_name: A name that can be used as a registry key for loading and saving dialog settings, such as its
                geometry and active selections.
            query: Interprocess communication object.
            parent: Parent widget for the dialog.
        """
        message = "This model control dialog is not currently implemented."
        message_with_ok(parent, message, 'Test', icon='Critical')

    @final
    def link_event(self, link_dict: dict[str, list[str]], lock_state: bool) -> MessagesAndRequests:
        """
        Handle when an item is being linked to the simulation.

        Args:
            link_dict: Dictionary with one item, whose key is the UUID of the item being linked, and whose value is
                a list of takes describing the item.
            lock_state: Whether the component is locked for editing.

        Returns:
            Messages and requests.
        """
        overridden = type(self).item_linked is not SimComponentBase.item_linked
        if not overridden:
            return [], []  # Don't launch a whole new process to run a method that does nothing.

        uuid, [unique_name] = link_dict.popitem()
        params = {'uuid': uuid, 'unique_name': unique_name, 'lock_state': lock_state}
        request = self._make_request(self._internal_link_event, params=params, needs_window=True)
        return [], [request]

    def _internal_link_event(self, query: Query, params: list[dict[str, Any]], parent: QWidget) -> MessagesAndRequests:
        """
        Check whether an item being linked is compatible with the simulation.

        Args:
            query: Interprocess communication object.
            params: List of one item, where item is a dict with one item, whose key is the UUID of the item being
                linked, and whose value is a list of takes describing the item.

        Returns:
            Messages and requests.
        """
        params = params.pop()
        uuid: str = params['uuid']
        unique_name: str = params['unique_name']
        lock_state: bool = params['lock_state']

        self.item_linked(query, uuid, unique_name, lock_state, parent)
        return self.messages, self.requests

    def item_linked(self, query: Query, linked_uuid: str, unique_name: str, lock_state: bool, parent: QWidget):
        """
        Handle when a new item was linked to the simulation.

        Args:
            query: Interprocess communication object.
            linked_uuid: UUID of the item that was just linked.
            unique_name: The unique-name of the item being linked, assuming the item's XML was designed the way this
                component expects.

                The sim component's `<component>` tag should have one `<takes>` tag inside for each type of thing that
                can be linked to the simulation. Each `<takes>` tag should have a `<declare_parameter>` tag inside it.
                The `unique_name` passed to this method will be the value of the `<declare_parameter>` tag for the item
                being linked. The value of this tag should probably be the item's unique-name (typically its class name)
                since it's basically the same thing.
            lock_state: Whether the item is currently locked for editing. Currently only makes sense in GMS.
            parent: Parent widget.
        """
        pass
