"""Base class for a simple model run tracker."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['ModelRunTracker']

# 1. Standard Python modules
from abc import ABC, abstractmethod

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv

# 4. Local modules
from xms.components.dmi.xms_data import XmsData


class ModelRunTracker(ABC):
    """
    A base class for a simple model run tracker.

    A model run tracker's main job is to tell XMS how close to completion the model run is. This usually involves
    observing the model's output in search of information about which time step it's on and comparing that to the
    expected total number of time steps.

    This base class is designed for such typical cases. It assumes the model runner will write all the model's output to
    `data.model_stdout_file` (or arrange for the model executable to do that itself) and does most of the work involved
    in dealing with the fact that that happens asynchronously. A derived class will typically override `process_line`
    to look at a line from the model's output and update `self.current_progress_percent` as necessary. The base class
    will take care of waiting until the `model_stdout_file` exists, stitching incomplete lines together, calling
    `self.process_line` exactly once with each line of model output, and informing XMS when `process_line` updates
    `self.current_progress_percent`.

    A potential weakness is that the stitching-incomplete-lines-together behavior assumes all lines end in newlines. If
    the model doesn't terminate its output with a final trailing newline, then the last line will never be processed.
    While the tracker could be changed to work around that, it would probably be model-specific, and more complicated
    than just having the model runner script write an extra newline to the output file after the model terminates.
    """
    def __init__(self, data: XmsData):
        """
        Initialize the tracker.

        Args:
            data: Interprocess communication object. If it was built with a real Query, that Query must have been
                initialized by passing `progress_script=True` to its constructor.
        """
        self.data = data

        self._current_progress_percent: float = 0.0
        self._ready_to_track: bool = False
        self._last_stdout_position = -1

    @property
    def current_progress_percent(self) -> float:
        """
        The current progress percent.

        Values should be a fraction from 0.0 (just started) to 1.0 (completely done).
        """
        return self._current_progress_percent

    @current_progress_percent.setter
    def current_progress_percent(self, value: float):
        """
        The current progress percent.

        Values should be a fraction from 0.0 (just started) to 1.0 (completely done).
        """
        assert 0.0 <= value <= 1.0
        assert value >= self._current_progress_percent
        if (value - self._current_progress_percent) >= 0.01:
            self._current_progress_percent = value

    @property
    def ready_to_track(self) -> bool:
        """
        Check if we're ready to start tracking.

        The model runner and progress tracker operate asynchronously. It's possible (though usually unlikely) that the
        first progress update will happen before the model actually writes anything.
        """
        if not self._ready_to_track:
            self._ready_to_track = self.data.model_stdout_file.exists()
        return self._ready_to_track

    def get_unread_lines(self) -> list[str]:
        """
        Get the lines of the model's stdout file that haven't been read yet.

        This method only gets complete lines. If the stdout file does not currently end in a newline, then the last line
        will be left for the next update on the assumption that the model wrote half a line and chose to buffer the rest
        for later. This means such behavior won't trip up the tracker, but that the tracker won't see the last line of
        output if the model doesn't write a newline. If the last line is important, the runner should write the missing
        newline after the model terminates.

        Each line in the returned list will end in a newline character.
        """
        lines = []

        with open(self.data.model_stdout_file, 'r') as f:
            if self._last_stdout_position > -1:
                f.seek(self._last_stdout_position)

            # `for line in f:` disables telling while iterating, so we have `iter()` do the iterating instead.
            for line in iter(f.readline, ''):
                if line.endswith('\n'):
                    self._last_stdout_position = f.tell()
                    lines.append(line)

        return lines

    def start_tracking(self):
        """Start the tracker."""
        self.data.start_progress_loop(self.try_update)

    def try_update(self):
        """Respond to a request to update the current progress state, wrapped in an exception handler."""
        try:
            self.update()
        except Exception as exc:
            XmEnv.report_error(exc)

    def update(self):
        """Respond to a request to update the current progress state."""
        if not self.ready_to_track:
            # We got our first update before the model wrote any output. Hopefully the next update will get something.
            return

        previous_progress = self.current_progress_percent

        for line in self.get_unread_lines():
            self.process_line(line.rstrip('\n'))

        if self.current_progress_percent != previous_progress:
            self.data.set_progress(self.current_progress_percent)

    @abstractmethod
    def process_line(self, line: str):
        """
        Process a line of the model's stdout.

        Args:
            line: The line to process. The newline on the end will be stripped.
        """
        # Implementations should look at the line and update self.current_progress_percent as necessary.
        # Incomplete lines will be postponed to the next update, so implementations don't have to worry about them.
        # They can safely assume every line passed to this function was originally followed by a newline.
        pass  # pragma: nocover
