from abc import ABC, abstractmethod
from enum import IntEnum
import uuid

from xms.grid.ugrid import UGrid


class GridType(IntEnum):
    unconstrained = 0
    ugrid_2d = 1
    ugrid_3d = 2
    rectilinear_2d = 3
    rectilinear_3d = 4
    quadtree_2d = 5
    quadtree_3d = 6


class Grid(ABC):
    """Class for representing a constrained UGrid."""

    def __init__(self, instance):
        """Only called by subclasses.
        Args:
            instance: The C++ wrapped instance.
        """
        if instance is None:
            raise ValueError('Grid must have an instance.')
        self._instance = instance

    @property
    @abstractmethod
    def grid_type(self):
        """GridType of the constraint."""
        pass

    def write_to_string(self):
        """Write the constraint to a readable string."""
        return self._instance.WriteToString()

    def write_to_file(self, file_path, binary_arrays=True):
        """Write the constraint to a file.
        Args:
            file_path (string): The file path to write to.
            binary_arrays (bool): Should binary arrays be written?
        """
        self._instance.WriteToFile(file_path, binary_arrays)

    @property
    def ugrid(self):
        """The constraint's UGrid."""
        return UGrid(instance=self._instance.GetXmUGrid())

    @property
    def uuid(self):
        """The constraint's UUID (or None)."""
        return self._instance.GetUuid()

    @uuid.setter
    def uuid(self, uuid):
        """The constraint's UUID."""
        valid = self._instance.SetUuid(uuid)
        if not valid:
            raise ValueError('Attempted to set invalid UUID.')
        self._instance.SetUuid(uuid)

    @property
    def point_elevations(self):
        """Get the Z values of the constraint's points."""
        return self._instance.GetPointElevations()

    @property
    def custom_point_elevations(self):
        """The constraint's custom point elevations. Needed for warping z
        location for rectilinear and quadtree
        constraints. """
        if not self._instance.HasPointElevations():
            return None
        return self._instance.GetPointElevations()

    @custom_point_elevations.setter
    def custom_point_elevations(self, elevations):
        """The constraint's custom point elevations. Needed for warping Z
        location for rectilinear and quadtree constraints. """
        if len(elevations) != self.ugrid.point_count:
            raise ValueError('Point elevation count must equal the number of '
                             'grid points.')
        self._instance.SetPointElevations(elevations)

    @property
    def cell_elevations(self):
        """The constraint's cell elevations."""
        if not self._instance.HasCellElevations():
            return None
        return self._instance.GetCellElevations()

    @cell_elevations.setter
    def cell_elevations(self, elevations):
        """The constraint's cell elevations."""
        if len(elevations) != self.ugrid.cell_count:
            raise ValueError('Elevation count must equal the number of grid '
                             'cells.')
        self._instance.SetCellElevations(elevations)

    def set_cell_tops_and_bottoms(self, tops, bottoms):
        """Set the constraint's cell tops and bottoms."""
        cell_count = self.ugrid.cell_count
        if len(tops) != cell_count or len(bottoms) != cell_count:
            raise ValueError('Count of tops and bottoms must equal the number '
                             'of grid cells.')
        self._instance.SetCellTopsAndBottoms(tops, bottoms)

    def has_cell_tops_and_bottoms(self):
        """Does the constraint have cell tops and bottoms?"""
        return self._instance.HasCellTopsAndBottoms()

    def delete_cell_tops_and_bottoms(self):
        """Deletes the cell tops and bottoms."""
        self._instance.DeleteCellTopsAndBottoms()

    def get_cell_tops(self):
        """Get the constraint's cell tops."""
        if not self._instance.HasCellTopsAndBottoms():
            return None
        return self._instance.GetCellTops()

    def get_cell_bottoms(self):
        """Get the constraint's cell bottoms."""
        if not self._instance.HasCellTopsAndBottoms():
            return None
        return self._instance.GetCellBottoms()

    @property
    def cell_layers(self):
        """The constraint's cell layers."""
        if not self._instance.HasCellLayers():
            return None
        return self._instance.GetCellLayers()

    @cell_layers.setter
    def cell_layers(self, layers):
        """The constraint's cell layers."""
        if len(layers) != self.ugrid.cell_count:
            raise ValueError('Layer count must equal the number of grid '
                             'cells.')
        self._instance.SetCellLayers(layers)

    @property
    def cell_centers(self):
        """The constraint's cell centers that were manually set on the grid."""
        if not self._instance.HasCellCenters():
            return None
        return self._instance.GetCellCenters()

    @cell_centers.setter
    def cell_centers(self, centers):
        """The constraint's cell centers that were manually set on the grid."""
        if len(centers) != self.ugrid.cell_count:
            raise ValueError('Centers count must equal the number of grid '
                             'cells.')
        self._instance.SetCellCenters(centers)

    @property
    def locked_xy_points(self):
        """If the constraint's points X/Y values are locked and shouldn't be
        edited."""
        return self._instance.GetLockedXyPoints()

    @locked_xy_points.setter
    def locked_xy_points(self, locked_xy_points):
        """If the constraint points X/Y values are locked and shouldn't be
        edited."""
        self._instance.SetLockedXyPoints(locked_xy_points)

    @property
    def model_on_off_cells(self):
        """If the constraint grid cells are turned on (1), off (0), or
        pass-through (-1). If result is an empty iterable they are all on."""
        return self._instance.GetModelOnOffCells()

    @model_on_off_cells.setter
    def model_on_off_cells(self, model_on_off_cells):
        """If the constraint grid cells are turned on (1), off (0), or
        pass-through (-1). Must be same size as number of cells."""
        if len(model_on_off_cells) != self.ugrid.cell_count:
            raise ValueError('Model on/off count must equal number of grid '
                             'cells.')
        self._instance.SetModelOnOffCells(model_on_off_cells)

    def check_all_cells_2d(self):
        """Check if all constraint cells are two dimensional."""
        return self._instance.CheckAllCells2d()

    def check_all_cells_3d(self):
        """Check if all constraint cells are three dimensional."""
        return self._instance.CheckAllCells3d()

    def check_correct_2d_edge_ordering(self):
        """Check if all adjacent cell edges have opposing edge direction."""
        return self._instance.CheckCorrect2dEdgeOrdering()

    def check_all_cells_are_of_type(self, cell_type):
        """Check if all the constraint's cells are of the given cell type.
        Args:
            cell_type: The UGrid cell type.
        """
        return self._instance.CheckAllCellsAreOfType(cell_type)

    def check_all_cells_are_of_types(self, cell_types):
        """Check if all the constraint's cells are one of the given cell types.
        Args:
            cell_types (iter): The UGrid cell types.
        """
        return self._instance.CheckAllCellsAreOfTypes(cell_types)

    def check_all_cells_are_same_square_in_xy(self):
        """Check if all of the constraint's cells are square. Only works for
        rectilinear and quadtree grids."""
        return self._instance.CheckAllCellsAreSameSquareInXy()

    def check_all_cells_are_rectangular_in_xy(self):
        """Check if all of the constraint's cells are rectangular. Only works
        for rectilinear and quadtree grids."""
        return self._instance.CheckAllCellsAreRectangularInXy()

    def check_has_disjoint_points(self):
        """Check if the constraint's grid has any disjoint points (not part of
        a cell)."""
        return self._instance.CheckHasDisjointPoints()

    def check_contiguous_cells_2d(self):
        """Check if the constraint's cells are all contiguously connected."""
        return self._instance.CheckContiguousCells2d()

    def check_has_2d_holes(self):
        """Check if the constraint's cells have any plan view holes (areas with
        no cells)."""
        return self._instance.CheckHas2dHoles()

    def check_all_cells_vertically_prismatic(self):
        """"Check if all of the constraint's cells are plan view vertically
        prismatic."""
        return self._instance.CheckAllCellsVerticallyPrismatic()

    def check_is_stacked_grid(self):
        """Check if the constraint's cells form stacked 3D layers.
        Each layer must have the same plan view topology."""
        return self._instance.CheckIsStackedGrid()

    def duplicate(self):
        """Get a hard copy of the constrained UGrid."""
        return self._instance.Duplicate()
