import uuid

from . import _xmsconstraint
from .grid import GridType
from .quadtree_grid_2d import QuadtreeGrid2d
from .quadtree_grid_3d import QuadtreeGrid3d
from .rectilinear_grid_2d import RectilinearGrid2d
from .rectilinear_grid_3d import RectilinearGrid3d
from .ugrid_2d import UGrid2d
from .ugrid_3d import UGrid3d
from .unconstrained_grid import UnconstrainedGrid


def _grid_from_cpp_co_grid(co_grid_types):
    """
    Get a wrapped Grid from a C++ wrapped CoGrid
    Args:
        co_grid_types: The C++ wrapped CoGrid.
    Returns:
        The python native wrapped Grid.
    """
    if co_grid_types.GetGridType() == GridType.quadtree_2d:
        return QuadtreeGrid2d(co_grid_types)
    if co_grid_types.GetGridType() == GridType.quadtree_3d:
        return QuadtreeGrid3d(co_grid_types)
    if co_grid_types.GetGridType() == GridType.rectilinear_2d:
        return RectilinearGrid2d(co_grid_types)
    if co_grid_types.GetGridType() == GridType.rectilinear_3d:
        return RectilinearGrid3d(co_grid_types)
    if co_grid_types.GetGridType() == GridType.ugrid_2d:
        return UGrid2d(instance=co_grid_types)
    if co_grid_types.GetGridType() == GridType.ugrid_3d:
        return UGrid3d(instance=co_grid_types)
    if co_grid_types.GetGridType() == GridType.unconstrained:
        return UnconstrainedGrid(instance=co_grid_types)
    raise ValueError('Unable to read CoGrid.')


def read_grid_from_string(string):
    """
    Read a constrained Grid from a string.
    Args:
        string: A string with Grid file contents.
    Returns:
        A constrained Grid.
    """
    co_grid_types = _xmsconstraint.constraint.coReadGridFromString(string)
    if co_grid_types is None:
        return None
    return _grid_from_cpp_co_grid(co_grid_types)


def read_grid_from_file(file_path):
    """
    Read a constrained Grid from a file.
    Args:
        file_path: Path to the file to read from.
    Returns:
        A constrained Grid.
    """
    co_grid_types = _xmsconstraint.constraint.coReadGridFromFile(file_path)
    if co_grid_types is None:
        return None
    return _grid_from_cpp_co_grid(co_grid_types)


def read_grid_uuid_from_string(s):
    """
    Read the UUID of a constrained Grid from a grid file.
    Args:
        s: Path to the file to read from.
    Returns:
        The UUID or None.
    """
    grid_uuid = _xmsconstraint.constraint.coReadUuidFromString(s)
    if grid_uuid:
        return grid_uuid
    return None


def read_grid_uuid_from_file(file_path):
    """
    Read the UUID of a constrained Grid from a grid file.
    Args:
        file_path: Path to the file to read from.
    Returns:
        The UUID or None.
    """
    grid_uuid = _xmsconstraint.constraint.coReadUuidFromFile(file_path)
    if grid_uuid:
        return grid_uuid
    return None


def copy_grid(co_grid):
    """Make a deep copy of a CoGrid.

    Args:
        co_grid (CoGrid): The python wrapped CoGrid to copy

    Returns:
        CoGrid: See description
    """
    new_grid = co_grid.duplicate()
    py_grid = _grid_from_cpp_co_grid(new_grid)
    py_grid.uuid = str(uuid.uuid4())
    return py_grid
