"""GenericCoverageComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import shutil
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, MenuItem
from xms.components.bases.coverage_component_base import CoverageComponentBase
from xms.components.display.display_options_io import read_display_options_from_json, write_display_options_to_json
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import Component
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList

# 4. Local modules

UNINITIALIZED_COMP_ID = -1
FEATURE_TYPE_TEXT = ['POINT', 'ARC', 'ARC_GROUP', 'POLYGON']  # Matches entity keywords from xmsapi


def get_component_data_object(main_file, comp_uuid, unique_name, name=''):
    """Create a data_object Component to send back to SMS to be built.

    Args:
        main_file (str): Path to the component main file
        comp_uuid: (str): UUID of the component
        unique_name (str): XML component unique name
        name (str): Tree item name of the component

    Returns:
        Component: data_object for the new component
    """
    return Component(
        name=name,
        comp_uuid=comp_uuid,
        main_file=main_file,
        model_name='Generic Coverages',
        unique_name=unique_name,
        locked=False
    )


def duplicate_display_opts(new_path, disp_opts_fname):
    """Duplicates display options.

    Args:
        new_path (str): Path to the new save location.
        disp_opts_fname (str): The filename (no path) of the display options JSON file

    Returns:
        (json_dict): dict containing the display options
    """
    fname = os.path.join(new_path, disp_opts_fname)
    json_dict = read_display_options_from_json(fname)
    if 'uuid' in json_dict:
        json_dict['uuid'] = str(uuid.uuid4())
        json_dict['comp_uuid'] = os.path.basename(new_path)
        categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
        categories.from_dict(json_dict)
        write_display_options_to_json(fname, categories)
    return json_dict


class GenericCoverageComponent(CoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base for the Generic Coverages model."""
    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file.strip('"\''))
        self.data = None
        self.tree_commands = []  # [(menu_text, menu_method)...]
        self.point_commands = []  # [(menu_text, menu_method)...]
        self.arc_commands = []  # [(menu_text, menu_method)...]
        self.arc_group_commands = []  # [(menu_text, menu_method)...]
        self.polygon_commands = []  # [(menu_text, menu_method)...]
        self.uuid = os.path.basename(os.path.dirname(self.main_file))
        self.disp_opts_files = []

    def _get_menu_action(self, command_method, id_files=None, selection=None, main_file=None):
        """Get an ActionRequest for a modal menu item command.

        Args:
            command_method (str): Name of the method to call
            id_files (dict): Paths to the XMS id and component id files, if applicable
            selection (dict): Feature ids of the selected items
            main_file (str): Path to the component main file.

        Returns:
            ActionRequest: The component menu item ActionRequest
        """
        main_file = main_file if main_file is not None else self.main_file
        action = ActionRequest(
            main_file=main_file,
            modality='MODAL',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name=command_method
        )
        parameters = {}
        if id_files is not None:
            parameters['id_files'] = id_files
        if selection is not None:
            parameters['selection'] = selection
        if parameters:
            action.action_parameters = parameters
        return action

    def _get_menu_item(self, command_text, command_method, id_files=None, selection=None, main_file=None):
        """Get a menu item for a modal command.

        Args:
            command_text (str): The menu text
            command_method (str): Name of the method to call
            id_files (dict): Paths to the XMS id and component id files, if applicable
            selection (dict): Feature ids of the selected items
            main_file (str): Path to the component main file.

        Returns:
            MenuItem: The component menu item
        """
        action = self._get_menu_action(command_method, id_files=id_files, selection=selection, main_file=main_file)
        return MenuItem(text=command_text, action=action)

    def _unpack_menu_id_files(self, id_files, selection, double_click):
        """Unpack the coverage component id files written by XMS for a right-click menu or double-click event.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC,
                ARC_GROUP, POLYGON). The value of the dictionary is a list of ids of the selected feature objects.
            id_files (dict): Key is entity type string, value is tuple of two str where first is the file location of
                the XMS coverage id binary file. Second is file location of the component coverage id binary file. Only
                applicable for coverage selections. File will be deleted after event. Copy if need to persist.
            double_click (bool): True if this is a double-click event on a selected feature object, False if this is
                a right-click menu event for a selected feature object. If true, only the first applicable menu command
                will be returned.

        Returns:
            menu_items (list[xms.api.dmi.MenuItem]): A list of menus and menu items to be shown. Note that this list can
            have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be added to the list
            to indicate a separator.
        """
        menu_list = []
        # Copy all the id files to a temporary location. XMS will delete them once this method returns.
        temp_dir = os.path.join(os.path.dirname(self.main_file), 'temp')
        os.makedirs(temp_dir, exist_ok=True)

        unpacked_id_files = {}
        for entity, filenames in id_files.items():
            if not os.path.exists(filenames[0]) or not os.path.exists(filenames[1]):
                continue
            temp_xms_file = os.path.join(temp_dir, os.path.basename(filenames[0]))
            temp_comp_file = os.path.join(temp_dir, os.path.basename(filenames[1]))
            io_util.copyfile(filenames[0], temp_xms_file)
            io_util.copyfile(filenames[1], temp_comp_file)
            unpacked_id_files[entity] = (temp_xms_file, temp_comp_file)

        all_commands = [self.point_commands, self.arc_commands, self.arc_group_commands, self.polygon_commands]
        found_one = False
        for commands, feature_type in zip(all_commands, FEATURE_TYPE_TEXT):
            if selection.get(feature_type):
                for command_text, command_method in commands:
                    menu_list.append(
                        self._get_menu_item(
                            command_text, command_method, id_files=unpacked_id_files, selection=selection
                        )
                    )
                    if double_click and found_one:
                        return menu_list  # Done if this is a double-click event
                    else:
                        found_one = True

        if not menu_list:
            shutil.rmtree(temp_dir, ignore_errors=True)  # Delete the id files if no menus were added.
        return menu_list

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            tuple: tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (list[tuple(str, str)]): List of tuples with the first element of the tuple being the message
                  level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
                - action_requests (list[xms.api.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        messages = []
        action_requests = []
        new_main_file = os.path.join(new_path, os.path.basename(self.main_file))
        return new_main_file, messages, action_requests

    def project_open_event(self, new_path):
        """Called when XMS project is opened.

        Components with display lists should add XmsDisplayMessage(s) to self.display_option_list
        """
        return  # This was a legacy thing, but we didn't have a release until this became unnecessary.

    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (list[str]): A list of the main files of the selected components of this type.

        Returns:
            menu_items (list[xms.api.dmi.MenuItem]): A list of menus and menu items to be shown. Note that this list can
            have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be added to the list
            to indicate a separator.
        """
        if len(main_file_list) > 1 or not main_file_list or not self.tree_commands:
            return []  # Multi-select, nothing selected, or no project explorer menu commands for this component

        menu_list = [None]  # None == spacer
        # Add all the project explorer menus
        for command_text, command_method in self.tree_commands:
            menu_list.append(self._get_menu_item(command_text, command_method, main_file=main_file_list[0][0]))
        return menu_list

    def get_double_click_actions(self, lock_state):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.

        Returns:
            tuple: tuple containing:
                - messages (list[tuple(str, str)]): List of tuples with the first element of the tuple being the message
                  level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
                - action_requests (list[xms.api.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        messages = []
        actions = []

        if self.tree_commands:  # If tree commands have been defined, the first will be the double-click action.
            actions.append(self._get_menu_action(self.tree_commands[0][1]))

        return messages, actions

    def get_display_menus(self, selection, lock_state, id_files):
        """This will be called when right-click menus in the main display area of XMS are being created.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC,
                ARC_GROUP, POLYGON). The value of the dictionary is a list of ids of the selected feature objects.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.
            id_files (dict): Key is entity type string, value is tuple of two str where first is the file location of
                the XMS coverage id binary file. Second is file location of the component coverage id binary file. Only
                applicable for coverage selections. File will be deleted after event. Copy if need to persist.

        Returns:
            menu_items (list[xms.api.dmi.MenuItem]): A list of menus and menu items to be shown. Note that this list can
            have objects of type xms.api.dmi.Menu as well as xms.api.dmi.MenuItem. "None" may be added to the list
            to indicate a separator.
        """
        return self._unpack_menu_id_files(id_files, selection, False)

    def get_double_click_actions_for_selection(self, selection, lock_state, id_files):
        """This will be called when a double-click in the main display area of XMS happened.

        Args:
            selection (dict): A dictionary with the key being a string of the feature entity type (POINT, ARC,
                ARC_GROUP, POLYGON). The value of the dictionary is a list of ids of the selected feature objects.
            lock_state (bool): True if the the component is locked for editing. Do not change the files if locked.
            id_files (dict): Key is entity type string, value is tuple of two str where first is the file location of
                the XMS coverage id binary file. Second is file location of the component coverage id binary file. Only
                applicable for coverage selections. File will be deleted after event. Copy if need to persist.

        Returns:
            tuple: tuple containing:
                - messages (list[tuple(str, str)]): List of tuples with the first element of the tuple being the message
                  level (DEBUG, ERROR, WARNING, INFO) and the second element being the message text.
                - action_requests (list[xms.api.dmi.ActionRequest]): List of actions for XMS to perform.
        """
        menus = self._unpack_menu_id_files(id_files, selection, True)
        actions = [menus[0].action_request] if menus else []
        return [], actions

    def get_display_options_action(self):
        """Get an ActionRequest that will refresh the XMS display list for components with display."""
        return ActionRequest(
            main_file=self.main_file,
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='get_initial_display_options',
            comp_uuid=self.uuid
        )
