"""Filter model for the observation target data table."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import Qt

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel


class ObsTargetModel(QxPandasTableModel):
    """A model to set header text and format time strings."""
    COL_TEXT = ['', '', 'Name', 'Time', 'Interval', 'Observed', 'Computed', 'Residual']
    COL_IDX_FEATURE_TYPE = 0
    COL_IDX_CATEGORY = 1  # CategoryDisplayOption.id from the display options JSON file
    COL_IDX_NAME = 2
    COL_IDX_TIME = 3
    COL_IDX_OBSERVED = 5
    COL_IDX_COMPUTED = 6
    COL_IDX_RESIDUAL = 7

    def __init__(self, data_frame, time_format, parent):
        """Initializes the filter model.

        Args:
            time_format (str): The absolute datetime format to use (should use strftime specifiers)
            data_frame (DataFrame): The DataFrame to use in the model
            parent (QWidget): The parent object.
        """
        self._add_residual_column(data_frame)
        super().__init__(data_frame, parent)
        self._data_frame = data_frame
        self._time_format = time_format

    def _add_residual_column(self, data_frame):
        """Add a column to the DataFrame for residual (observed - computed).

        Args:
            data_frame (DataFrame): The DataFrame to use in the model
        """
        data_frame['residual'] = data_frame['observed'] - data_frame['computed']

    def headerData(self, section, orientation, role=Qt.DisplayRole):  # noqa: N802
        """Returns the data for the given role and section in the header.

        Args:
            section (int): The section.
            orientation (:obj:`Qt.Orientation`): The orientation.
            role (int): The role.

        Returns:
            The data.
        """
        if role != Qt.DisplayRole or section < 0:
            return super().headerData(section, orientation, role)

        if orientation == Qt.Vertical:  # 1-base sequential row number
            return str(section + 1)
        else:  # Pretty text for the column header
            return self.COL_TEXT[section]

    def data(self, index, role=Qt.DisplayRole):
        """Depending on the index and role given, return data, or None.

        Args:
            index (:obj:`QModelIndex`): The index.
            role (int): The role.

        Returns:
            The data at index, or None.
        """
        if role in [Qt.DisplayRole, Qt.EditRole]:
            col_idx = index.column()
            val = self._data_frame.iat[index.row(), col_idx]
            if col_idx == self.COL_IDX_TIME:  # Format the time
                if isinstance(val, pd.Timestamp):  # Format datetimes based on current SMS preferences.
                    return pd.to_datetime(val).strftime(self._time_format)
                return str(val)  # Float offset
            val = str(val)
            if val in ['nan', 'None']:
                return ''  # Show missing data as blank cells in the spreadsheet
            return val
        return super().data(index, role)
