"""Methods to ensure a polygon is oriented correctly."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import FilterLocation

# 4. Local modules


def get_poly_points(arc_list, arc_directions, inner=False):
    """Returns a list of the polygon's x,y,z points.

    Last point is a repeat of the first point.

    Args:
        arc_list (list of data_objects.parametrs.Arc): List of arcs that make up the polygon.
        arc_directions (list of data_objects.parameters.BoolLiteral): List of arc directions.
        inner (bool): True if this is an interior arc

    Returns:
        (list of tuple of float): See description.
    """
    poly = []
    for i, arc in enumerate(arc_list):
        arc_points = arc.get_points(FilterLocation.PT_LOC_ALL)
        reverse = (inner and not arc_directions[i]) or (not inner and arc_directions[i])
        if reverse:  # reverse it
            arc_xyz_list = [(point.x, point.y, point.z) for point in reversed(arc_points)]
        else:
            arc_xyz_list = [(point.x, point.y, point.z) for point in arc_points]
        poly.extend(arc_xyz_list[:-1])
    poly.append(poly[0])  # Repeat the first point
    return poly


def get_polygon_point_lists(do_polygon):
    """Returns a list of lists of points defining a polygon with possible inner holes.

    Last points are repeats of the first points.

    Args:
        do_polygon (xms.data_objects.parameters.Polygon):

    Returns:
        (list of lists of tuple): The lists of points defining the polygon.
    """
    polygon = []

    # Outer arcs
    poly = get_poly_points(do_polygon.arcs, do_polygon.arc_directions)
    make_poly_clockwise(poly)
    polygon.append(poly)

    # Interior polygons
    interior_arcs_list = do_polygon.interior_arcs
    arc_directions_list = do_polygon.interior_arc_directions
    for inner_arcs, arc_directions in zip(interior_arcs_list, arc_directions_list):
        poly = get_poly_points(inner_arcs, arc_directions, True)
        make_poly_counter_clockwise(poly)
        polygon.append(poly)
    return polygon


def make_poly_clockwise(poly):
    """Makes the polygon clockwise.

    Last point expected to be repeat of first point.

    Args:
        poly (list[tuple]): List of xyz points, e.g. [(1,2,3), (4,5,6)]
    """
    area = poly_area_x2(poly)
    if area > 0.0:
        poly.reverse()


def make_poly_counter_clockwise(poly):
    """Makes the polygon clockwise.

    Last point expected to be repeat of first point.

    Args:
        poly (list[tuple]): List of xyz points, e.g. [(1,2,3), (4,5,6), (1,2,3)]
    """
    area = poly_area_x2(poly)
    if area < 0.0:
        poly.reverse()


def poly_area_x2(poly):
    """Computes 2 times the area of the poly (2 times because its faster).

    Last point expected to be repeat of first point.
    Positive area if counter clockwise polygon, otherwise area is negative.

    This doesn't include a translation to the origin to improve accuracy as
    we don't need it to be that accurate.

    Args:
        poly: A poly (list of point indexes).

    Returns:
        The area.
    """
    if len(poly) < 3:
        return 0.0

    area = 0.0
    for i in range(len(poly) - 1):
        area += (poly[i][0] * poly[i + 1][1])
        area -= (poly[i][1] * poly[i + 1][0])
    return area
