"""Utility methods for running tests with coverages."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
import warnings

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.display.display_options_io import read_display_options_from_json
from xms.data_objects.parameters import Coverage, FilterLocation
from xms.testing import file_comparison

# 4. Local modules


def export_coverage_to_ascii(filename: str | Path, coverage: Coverage):
    """Exports the coverage to filename as ASCII to help in debugging.

    Args:
        filename: Filepath to ASCII file.
        coverage: The coverage.
    """
    with open(filename, 'w') as file:
        file.write(f'Name: {coverage.name}\n')

        # Points
        disjoint_pts = coverage.get_points(FilterLocation.PT_LOC_DISJOINT)
        file.write('Disjoint Points\n')
        file.write('ID\tX\tY\tZ\n')
        for pt in disjoint_pts:
            file.write(f'{pt.id}\t{pt.x}\t{pt.y}\t{pt.z}\n')

        # Nodes
        nodes = coverage.get_points(FilterLocation.PT_LOC_CORNER)
        file.write('\nNodes\n')
        file.write('ID\tX\tY\tZ\n')
        for pt in nodes:
            file.write(f'{pt.id}\t{pt.x}\t{pt.y}\t{pt.z}\n')

        # Arcs
        file.write('\nArcs\n')
        for arc in coverage.arcs:
            file.write(f'Arc ID: {arc.id}\t')
            file.write(f'Node1 ID: {arc.start_node.id}\t')
            file.write(f'Node2 ID: {arc.end_node.id}\n')
            file.write('Vertices\n')
            file.write('X\tY\tZ\n')
            for pt in arc.vertices:
                file.write(f'{pt.x}\t{pt.y}\t{pt.z}\n')

        # Polygons
        file.write('\nPolygons\n')
        for poly in coverage.polygons:
            file.write(f'Polygon ID: {poly.id}\n')
            file.write(f'Outer arcs: {" ".join([str(arc.id) for arc in poly.arcs])}\n')
            for i, inner_poly in enumerate(poly.interior_arcs):
                file.write(f'Inner poly {i}: {" ".join([str(arc.id) for arc in inner_poly])}\n')


def compare_ascii_files(baseline, output, ignore_tokens=None):
    """Compare to ascii files, optionally ignoring certain tokens.

    Args:
        baseline (str): Filename of the baseline
        output (str): Filename of the file to compare to the baseline
        ignore_tokens (list): List of tokens. If token is found on a line, those lines are not compared

    Returns:
        bool: True if the files are equal
    """
    msg = 'This function is deprecated. Use xms.testing.file_comparison.ascii_files_equal() instead.'
    warnings.warn(msg, DeprecationWarning, stacklevel=2)
    return file_comparison.ascii_files_equal(baseline, output, comments=ignore_tokens)


def compare_display_options_json(baseline, output):
    """Compare to display options JSON files.

    Note that this will set the 'uuid' and 'comp_uuid' variables to empty string before comparing. You will need to
    check those manually if you want to ensure they match.

    Args:
        baseline (str): Filename of the baseline
        output (str): Filename of the file to compare to the baseline

    Returns:
        bool: True if the files are equal
    """
    baseline_dict = read_display_options_from_json(baseline)
    output_dict = read_display_options_from_json(output)
    baseline_dict['uuid'] = ''
    baseline_dict['comp_uuid'] = ''
    output_dict['uuid'] = ''
    output_dict['comp_uuid'] = ''
    return baseline_dict == output_dict


def export_map_file(file_path: str | Path, coverages: list[Coverage], write_uuid: bool = False) -> None:
    """Exports a .map file given the list of coverages.

    Args:
        file_path: File path to .map file that will be created.
        coverages: List of coverages.
        write_uuid: If True, the coverage uuid is written.
    """
    with open(file_path, 'w') as file:
        file.write('MAP\n')
        file.write('VERSION 5.0\n')
        for coverage in coverages:
            file.write('BEGCOV\n')
            file.write(f'COVNAME "{coverage.name}"\n')
            if write_uuid:
                file.write(f'COVGUID "{coverage.uuid}"\n')

            # Points
            disjoint_pts = coverage.get_points(FilterLocation.PT_LOC_DISJOINT)
            for pt in disjoint_pts:
                file.write('POINT\n')
                file.write(f'XY {pt.x} {pt.y} {pt.z}\n')
                file.write(f'ID {pt.id}\n')
                file.write('END\n')

            # Nodes
            nodes = coverage.get_points(FilterLocation.PT_LOC_CORNER)
            for pt in nodes:
                file.write('NODE\n')
                file.write(f'XY {pt.x} {pt.y} {pt.z}\n')
                file.write(f'ID {pt.id}\n')
                file.write('END\n')

            # Arcs
            arcs = coverage.arcs
            for arc in arcs:
                file.write('ARC\n')
                file.write(f'ID {abs(arc.id)}\n')
                file.write(f'NODES {arc.start_node.id} {arc.end_node.id}\n')
                verts = arc.vertices
                file.write(f'ARCVERTICES {len(verts)}\n')
                for v in verts:
                    file.write(f'{v.x} {v.y} {v.z}\n')
                file.write('END\n')

            # Polygons
            polys = coverage.polygons
            if polys and polys[0].arcs:
                # XMS unconditionally throws out the first polygon, which means it won't read this coverage correctly
                # if we write its polygons as-is. We insert a bogus universal polygon first here so SMS has something to
                # throw away and the rest of them will be preserved.
                file.write('POLYGON\n')
                file.write('ID 0\n')
                file.write('ARCS 0\n')
                file.write('END\n')
            for poly in polys:
                file.write('POLYGON\n')
                file.write(f'ID {poly.id}\n')
                out_arcs = poly.arcs
                file.write(f'ARCS {len(out_arcs)}\n')
                for arc in out_arcs:
                    file.write(f'{arc.id}\n')
                for inner_poly in poly.interior_arcs:
                    file.write(f'HARCS {len(inner_poly)}\n')
                    for arc in inner_poly:
                        file.write(f'{arc.id}\n')
                file.write('END\n')
            file.write('ENDCOV\n')


def export_coverage_to_map(filename: str | Path, coverage: Coverage, write_uuid: bool = False):
    """Exports the coverage to filename as xms map file to help in debugging.

    Args:
        filename: Filepath to map file.
        coverage: The coverage.
        write_uuid: If True, the coverage uuid is written.
    """
    export_map_file(filename, [coverage], write_uuid)
