"""Module for importing a CSTORM simulation with feedback."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.components.display import windows_gui as win_gui
from xms.guipy.dialogs import xms_parent_dlg as xms_dlg
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules
from xms.cstorm.file_io import util
from xms.cstorm.file_io.control_reader import ControlReader

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"


class ImportSimulationRunner(ProcessFeedbackThread):
    """Class for importing a CSTORM simulation in a worker thread."""
    def __init__(self, parent):
        """Construct the worker.

        Args:
            parent (:obj:`QWidget`): Qt parent
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._logger = util.get_logger()
        self.query = Query()
        self.filename = self.query.read_file
        self.reader = None

    def _do_work(self):
        """Thread worker method."""
        try:
            self.reader = ControlReader(self.filename, self.query)
            self.reader.read()
        except Exception:
            self._logger.exception('Error importing simulation.')


def import_simulation_with_feedback():
    """Export a simulation with a feedback dialog."""
    xms_dlg.ensure_qapplication_exists()
    # Parse parent HWNDs and icon path from commandline arguments.
    if XmEnv.xms_environ_running_tests() != 'TRUE':
        parent_hwnd, main_hwnd, _ = xms_dlg.parse_parent_window_command_args()
        parent = xms_dlg.get_parent_window_container(parent_hwnd)
        # Create the timer that keeps our Python dialog in the foreground of XMS.
        _ = win_gui.create_and_connect_raise_timer(main_hwnd, parent)  # Keep the timer in scope
    else:
        parent = None
        main_hwnd = None

    worker = ImportSimulationRunner(parent)
    # from xms.guipy import debug_pause
    # debug_pause()
    # worker.do_work()
    # return

    display_text = {
        'title': 'CSTORM Import Simulation',
        'working_prompt': 'Importing CSTORM simulation files. Please wait...',
        'warning_prompt': 'Warning(s) encountered while importing simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered while importing simulation. Review log output for more details.',
        'success_prompt': 'Successfully imported simulation',
        'note': '',
        'auto_load': 'Close this dialog automatically when importing is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms.cstorm', worker=worker, parent=None)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():
        if parent is not None:
            # Bring top-level Python window to foreground. Do this before send() so we immediately give focus
            # back to SMS.
            win_gui.raise_main_xms_window(main_hwnd)
        worker.query.send()  # Send data back to XMS
