"""Python wrapping for xms._data_objects.parameters.Coverage."""
# 1. Standard python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.data_objects._data_objects.parameters import Coverage as CCoverage
from xms.data_objects.parameters.spatial.Arc import Arc as PyArc
from xms.data_objects.parameters.spatial.Point import Point as PyPoint
from xms.data_objects.parameters.spatial.Polygon import Polygon as PyPolygon
from xms.data_objects.parameters.spatial.spatial_vector.SpatialVector import SpatialVector


XMDF_DEFAULT_COVERAGE_PATH = 'Map Data/Coverage1'  # Matches constant defined in C++ library


class Coverage(SpatialVector):
    """The pure Python wrapper for C++ exposed xms._data_objects.parameters.Coverage objects."""
    def __init__(self, *args, **kwargs):
        """Construct the wrapper.

        Args:
            *args:
                filename (str, optional): The path to the H5 file containing the coverage data (in internal format)
                h5_group (str, optional): Path in the H5 file to the coverage data. XMDF_DEFAULT_COVERAGE_PATH by
                    default.
            **kwargs:
                'instance': CCoverage - The C++ Coverage object to wrap
                See SpatialVector for base class kwargs
        """
        num_args = len(args)
        if num_args > 0:  # Use H5 file constructor
            group_path = XMDF_DEFAULT_COVERAGE_PATH if num_args == 1 else args[1]
            kwargs['instance'] = CCoverage(args[0], group_path)
        elif 'instance' not in kwargs:  # Default empty coverage
            kwargs['instance'] = CCoverage()
        super().__init__(**kwargs)

    @property
    def polygons(self):
        """Returns a list of the coverage polygons as a PyPolygon."""
        polys = self._instance.GetPolygons()
        # Create a pure Python wrappers for the polygons.
        return [PyPolygon(instance=poly) for poly in polys]

    @polygons.setter
    def polygons(self, polys):
        """Sets the coverage polygons.

        Args:
            polys (list of PyPolygon): The coverage polygons
        """
        # Unwrap pure Python polygons
        self._instance.SetPolygons([poly._instance for poly in polys])

    @property
    def arcs(self):
        """Returns a list of the coverage arcs as a PyArc."""
        arcs = self._instance.GetArcs()
        # Create a pure Python wrappers for the arcs.
        return [PyArc(instance=arc) for arc in arcs]

    @arcs.setter
    def arcs(self, cov_arcs):
        """Sets the coverage arcs.

        Args:
            cov_arcs (list of PyArc): The coverage arcs
        """
        # Unwrap pure Python arcs
        self._instance.SetArcs([arc._instance for arc in cov_arcs])

    @property
    def arc_groups(self):
        """Returns a mapping of the arc group IDs to a list of the arc IDs in that group."""
        return self._instance.GetArcGroups()

    @arc_groups.setter
    def arc_groups(self, groups):
        """Sets the coverage arc groups.

        Notes:
            This will overwrite any previously defined arc groups, so you should only call this once if building a
            coverage to send back to XMS

        Args:
            groups (dict[int, list[int]]): Mapping of the arc group IDs to a list of the arc IDs in that group
        """
        self._instance.SetArcGroups(groups)

    @property
    def filename_and_group(self):
        """Get the H5 dump file name.

        Notes:
            This file is deleted as soon as coverage is read into memory, so copy it before calling any other accessors.

        Returns:
            tuple[str, str]: The path to the H5 dump file, group path inside file where coverage is written
        """
        return self._instance.GetDumpFile()

    def get_points(self, location):
        """Gets the coverage points at a specified location.

        Args:
            location (xms.data_objects.parameters.FilterLocation): The point locations to retrieve. Valid values:
                FilterLocation.PT_LOC_DISJOINT - disjoint coverage points
                FilterLocation.PT_LOC_CORNER - arc start/end nodes
                FilterLocation.PT_LOC_MID - arc vertices
                FilterLocation.ARC_LOC_CENTER - polygon centroids (may or may not be defined)
                FilterLocation.PT_LOC_ALL - all points in the coverage

                Note that the FilterLocation enum supports bitwise OR operations.
        Returns:
            list of PyPoint: The coverage points at the specified location.
        """
        points = self._instance.GetPoints(location)
        return [PyPoint(instance=pt) for pt in points]

    def set_points(self, points):
        """Sets the coverage points.

        Args:
            points (list of PyPoint): The coverage points
        """
        # Unwrap pure Python arcs
        self._instance.SetPoints([pt._instance for pt in points])

    def write_h5(self, filename, group_path=XMDF_DEFAULT_COVERAGE_PATH, overwrite=True):
        """Write the coverage data to an H5 file in internal format.

        Args:
            filename (str or xms.data_objects._data_objects.parameters.FileLocation): Path to the output H5 file. Either
                a string or a FileLocation object retrieved from an xms.api.dmi.Query.
            group_path (str): Group path in the H5 file of the coverage data
            overwrite (bool): If True, overwrite any existing data in the file
        """
        self.complete()  # Ensure complete flag is set to True.
        self._instance.InitH5(filename, group_path, overwrite)
