"""Class that stores data set meta data and converts to/from a dict."""

# 1. Standard Python modules
from typing import Dict, Generator, List, Tuple, Union

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules

XMDF_DATA_LOCATIONS = {
    'cells': 0,
    'points': 1,
    'faces': 2,
    'face_i': 3,
    'face_j': 4,
    'face_k': 5,
    'column': 6,
    'center_corner_faces': 7,
}


class DatasetMetadata:
    """A class that stores metadata for a dataset.

    Currently, this class only contains the minimal amount of data needed to replace UGrid values such as cell top,
    bottom, and model on/off cells. Could be expanded to replace the use of XMDF for sending datasets from a script.
    See XmsDmiUGridCallback::UpdateCellTopAndBottomElevations in XMS for how array values are parsed.

    """
    def __init__(self):
        """Construct the metadata storage."""
        self.geom_uuid = ''
        self.value_files = []  # List of data value files for dataset. Multiple if vector or UGrid tops and bottoms

    def from_dict(self, metadata_dict: Dict) -> None:
        """Populate metadata object from dictionary.

        Args:
            metadata_dict (:obj:`dict`): Metadata dict. Should be in format used to read/write metadata to json with
                orjson

        """
        if 'geom_uuid' in metadata_dict:
            self.geom_uuid = metadata_dict['geom_uuid']
        if 'value_files' in metadata_dict:
            for file in metadata_dict['value_files']:
                self.value_files.append(file)

    def __iter__(self) -> Generator[Tuple[str, Union[str, List[str]]], None, None]:
        """Make the metadata iterable."""
        yield 'geom_uuid', self.geom_uuid
        yield 'value_files', self.value_files
