"""Helper methods for manipulating vector datasets."""

# 1. Standard Python modules
from typing import Optional, Sequence, Tuple

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules

# 4. Local modules
from xms.datasets.numpy_util import ensure_sequence_is_numpy_array


def find_vector_mins_and_maxs(data_cube: Sequence) -> Tuple[np.ndarray, np.ndarray]:
    """Compute mins and maxs of n-number of timesteps.

    Args:
        data_cube (:obj:`Sequence`): The vector dataset. Data dimensions = (num_times, num_values, 2)

    Returns:
        (:obj:`tuple(numpy.ndarray, numpy.ndarray)`):
            0 = The minimum vector magnitudes for each timestep of the dataset
            1 = The maximum vector magnitudes for each timestep of the dataset
    """
    # Compute the vector magnitude min/max values at each time step.
    mags = vx_vy_to_magnitude(vectors=data_cube)
    mins = np.nanmin(mags, axis=1)
    maxs = np.nanmax(mags, axis=1)
    return mins, maxs


def vx_vy_to_direction(vx: Sequence, vy: Sequence) -> np.ndarray:
    """Compute a direction dataset from x/y vector components.

    Works with a single timestep or an entire dataset.

    Args:
        vx (:obj:`Sequence`): The vector x components
        vy (:obj:`Sequence`): The vector y components

    Returns:
        (:obj:`numpy.ndarray`): The vector directions in degrees. Data has same shape as vx/vy.
    """
    directions = np.degrees(np.arctan2(vy, vx))
    directions[directions < 0.0] += 360.0
    # Turn Vx=0.0, Vy=0.0 to 0.0 directions? That is what SMS does.
    directions = np.where(np.logical_and(vx == 0.0, vy == 0.0), 0.0, directions)
    return directions


def vx_vy_to_magnitude(
    vx: Optional[Sequence] = None, vy: Optional[Sequence] = None, vectors: Optional[Sequence] = None
) -> np.ndarray:
    """Compute a magnitude dataset from x/y vector components or a vector dataset.

    Args:
        vx (:obj:`Sequence`): The vector x components. Mutually exclusive with vectors.
        vy (:obj:`Sequence`): The vector y components. Mutually exclusive with vectors.
        vectors (:obj:`Sequence`): The vector x,y components. Mutually exclusive with vx and vy. Dimensions must be:
            single timestep = (num_values, 2) or n-number timesteps = (num_times, num_values, 2)

    Returns:
        (:obj:`numpy.ndarray`): The vector magnitudes. Data has same shape as vx/vy. If input passed as a vector
        dataset, returned dataset is scalar with same time and value dimensions.
    """
    if vectors is None:
        if vx is None or vy is None:
            raise ValueError('Must provide vx and vy or vectors')
        vectors = np.vstack((vx, vy)).T
    vectors = ensure_sequence_is_numpy_array(vectors)
    return np.linalg.norm(vectors, axis=len(vectors.shape) - 1)


def magdir_to_vx_vy(magnitudes: Sequence, directions: Sequence) -> Tuple[np.ndarray, np.ndarray]:
    """Compute a direction dataset from x/y vector components.

    Works with a single timestep or an entire dataset.

    Args:
        magnitudes (:obj:`Sequence`): The vector magnitudes
        directions (:obj:`Sequence`): The vector directions

    Returns:
        tuple(numpy.ndarray, numpy.ndarray): The Vx and Vy components
    """
    # Ensure we have magnitudes in a numpy array.
    magnitudes = ensure_sequence_is_numpy_array(magnitudes)
    vx = magnitudes * np.cos(np.radians(directions))
    vy = magnitudes * np.sin(np.radians(directions))
    return vx, vy
