"""Code to handle XMS DMI component display events for EwnCoverageComponent."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid

# 2. Third party modules
import numpy as np
from PySide2.QtGui import QColor
from PySide2.QtWidgets import QDialog
import xarray as xr

# 3. Aquaveo modules
import xms.api.dmi as xmd
import xms.api.dmi.XmsEnvironment as Xme
from xms.components.display.display_options_io import (
    read_display_options_from_json, write_display_option_ids, write_display_options_to_json
)
from xms.components.display.xms_display_message import XmsDisplayMessage
from xms.grid.geometry import geometry as geoms
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.polygon_texture import PolygonOptions, PolygonTexture
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.ewn.data import ewn_cov_data_consts as ewn_consts
from xms.ewn.dmi.ewn_component_queries import add_output_ugrid, add_transition_poly_coverage
from xms.ewn.gui.assign_ewn_arc_dialog import AssignEwnArcDialog
from xms.ewn.gui.assign_ewn_polygon_dialog import AssignEwnPolygonDialog
from xms.ewn.gui.assign_sediment_volume_dialog import AssignSedimentVolumeDialog
from xms.ewn.gui.dredge_calculator_dialog import DredgeCalculatorDialog
from xms.ewn.gui.ewn_properties_dialog import EwnPropertiesDialog
from xms.ewn.gui.insert_levee_dialog import InsertLeveeDialog
from xms.ewn.gui.refine_quadtree_dialog import RefineQuadTreeDialog
from xms.ewn.gui.sediment_properties_dialog import SedimentPropertiesDialog
from xms.ewn.gui.tool_input_dialog import ToolInputDialog
from xms.ewn.gui.update_adh_friction_dialog import UpdateAdhFrictionDialog

EWN_DISPLAY_OPTIONS_JSON = 'ewn_cov_display_options.json'
SEDIMENT_DISPLAY_OPTIONS_JSON = 'sediment_display_options.json'


def duplicate_display_opts(new_path, sediment=False):
    """Duplicates display options.

    Args:
        new_path (:obj:`str`): Path to the new save location.
        sediment (:obj:`bool`): True if duplicating a sediment management coverage's display options

    Returns:
        (:obj:`json_dict`): dict containing the display options
    """
    display_json = SEDIMENT_DISPLAY_OPTIONS_JSON if sediment else EWN_DISPLAY_OPTIONS_JSON
    fname = os.path.join(new_path, display_json)
    json_dict = read_display_options_from_json(fname)  # Read the old display options
    # Reset the display list and component UUIDs.
    json_dict['comp_uuid'] = os.path.basename(new_path)
    json_dict['uuid'] = str(uuid.uuid4())
    categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
    categories.from_dict(json_dict)
    write_display_options_to_json(fname, categories)
    return json_dict


class EwnCoverageDisplay:
    """Helper class to manage EWN Coverage component display."""
    def __init__(self, enw_comp, has_display=True):
        """Constuct the helper.

        Args:
            enw_comp (EwnCoverageComponent): The EWN coverage component
            has_display (:obj:`bool`): flag to indicate that the component has display properties
        """
        self._ewn_comp = enw_comp
        self._sel_poly_ids = []
        self._sel_poly_comp_ids = set()
        self._sel_arc_ids = []
        self._sel_arc_comp_ids = set()
        self._pe_tree = None
        self._has_display = has_display

    def _get_selected_poly_comp_ids(self):
        """Gets the component ids for the selected arcs."""
        for xms_id in self._sel_poly_ids:
            comp_id = self._ewn_comp.get_comp_id(TargetType.polygon, xms_id)
            self._sel_poly_comp_ids.add(comp_id)

    def _get_selected_arc_comp_ids(self):
        """Gets the component ids for the selected arcs."""
        for xms_id in self._sel_arc_ids:
            comp_id = self._ewn_comp.get_comp_id(TargetType.arc, xms_id)
            self._sel_arc_comp_ids.add(comp_id)

    def _update_sediment_id_files(self):
        """Writes the display id files for the sediment volume management coverage."""
        for idx, _ in ewn_consts.EWN_POLYGON_SEDIMENT_TYPES.items():
            type_dset = self._ewn_comp.data.polygons.where(  # Select by feature type
                self._ewn_comp.data.polygons.sediment_type == idx,
                drop=True)
            comp_ids = type_dset.comp_id.data.astype(np.int32).tolist()
            id_file = os.path.join(os.path.dirname(self._ewn_comp.main_file), f'sediment_{idx}.ids')
            write_display_option_ids(id_file, comp_ids)

    def _update_poly_id_files(self):
        """Writes the display id files."""
        df = self._ewn_comp.data.meta_data
        cats = ['Unassigned'] + df['Feature type/Region'].tolist()
        for idx, _ in enumerate(cats):
            type_dset = self._ewn_comp.data.polygons.where(  # Select by feature type
                self._ewn_comp.data.polygons.classification == idx,
                drop=True)
            comp_ids = type_dset.comp_id.data.astype(np.int32).tolist()
            id_file = os.path.join(os.path.dirname(self._ewn_comp.main_file), f'cat_{idx}.ids')
            write_display_option_ids(id_file, comp_ids)

    def _update_poly_component(self, poly_data):
        """Update component data and display when dialog is accepted.

        Args:
            poly_data (:obj:`xarray.Dataset`): The updated polygon attributes
        """
        # Remove all data from the Dataset associated with component ids that are no longer used.
        self._clean_up_unused_polygon_comp_ids()

        # Set the component ids of all selected polygons in XMS to the newly generated component id.
        for poly_id in self._sel_poly_ids:
            # Add the dialog data to the Dataset at a newly generated comp_id coord for each selected polygon.
            new_comp_id = self._ewn_comp.data.add_polygon(poly_data)
            self._ewn_comp.update_component_id(TargetType.polygon, poly_id, new_comp_id)

        # Update component display id files and request a display refresh in XMS.
        if self._has_display:
            if self._ewn_comp.data.info.attrs['FILE_TYPE'] == 'EWN_SEDIMENT_MANAGEMENT_DATA':
                self._update_sediment_id_files()
            else:
                self._update_poly_id_files()
            self._ewn_comp.display_option_list.append(
                XmsDisplayMessage(file=self._ewn_comp.disp_opts_file, edit_uuid=self._ewn_comp.cov_uuid)
            )

        self._ewn_comp.data.commit()  # Save changes to disk.

    def _update_arc_component(self, arc_data):
        """Update component data and display when dialog is accepted.

        Args:
            arc_data (:obj:`xarray.Dataset`): The updated arc attributes
        """
        # Remove all data from the Dataset associated with component ids that are no longer used.
        self._clean_up_unused_arc_comp_ids()

        # Set the component ids of all selected polygons in XMS to the newly generated component id.
        for arc_id in self._sel_arc_ids:
            # Add the dialog data to the Dataset at a newly generated comp_id coord for each selected polygon.
            new_comp_id = self._ewn_comp.data.add_arc(arc_data)
            self._ewn_comp.update_component_id(TargetType.arc, arc_id, new_comp_id)

        self._ewn_comp.data.commit()

    def _clean_up_unused_polygon_comp_ids(self):
        """Remove unused comp ids from the polygon data."""
        selected_poly_ids = set(self._sel_poly_ids)
        delete_comp_ids = set()
        comp_ids = self._ewn_comp.data.polygons.comp_id.data.astype(np.int32).tolist()
        for cid in comp_ids:
            xms_ids = self._ewn_comp.get_xms_ids(TargetType.polygon, cid)
            if type(xms_ids) is int:
                delete_comp_ids.add(cid)
            elif cid in self._sel_poly_comp_ids:  # One or more selected polygons previously had this component id.
                # If all the polygons using this component id are selected, delete it. We will assign a new
                # component id to all those polygons, so the old component id is no longer being used.
                reassigned_polys = set(xms_ids) & selected_poly_ids
                if len(reassigned_polys) == len(xms_ids):
                    delete_comp_ids.add(cid)
        self._ewn_comp.data.remove_poly_comp_ids(delete_comp_ids)

    def _clean_up_unused_arc_comp_ids(self):
        """Remove unused comp ids from the arc data."""
        selected_arc_ids = set(self._sel_arc_ids)
        delete_comp_ids = set()
        comp_ids = self._ewn_comp.data.arcs.comp_id.data.astype(np.int32).tolist()
        for cid in comp_ids:
            xms_ids = self._ewn_comp.get_xms_ids(TargetType.arc, cid)
            if type(xms_ids) is int:
                delete_comp_ids.add(cid)
            elif cid in self._sel_arc_comp_ids:
                reassigned_arcs = set(xms_ids) & selected_arc_ids
                if len(reassigned_arcs) == len(xms_ids):
                    delete_comp_ids.add(cid)
        self._ewn_comp.data.remove_arc_comp_ids(delete_comp_ids)

    def get_initialize_display_action(self):
        """Get an ActionRequest to call get_initial_display_options."""
        action = xmd.ActionRequest(
            main_file=self._ewn_comp.main_file,
            modality='NO_DIALOG',
            class_name=self._ewn_comp.class_name,
            module_name=self._ewn_comp.module_name,
            method_name='get_initial_display_options',
            comp_uuid=self._ewn_comp.uuid
        )
        return action

    def initialize_display(self, query):
        """Handle the get_initial_display_options event.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
        """
        if query is not None:
            self._ewn_comp.cov_uuid = query.parent_item_uuid()
            if not self._ewn_comp.cov_uuid:
                return [('ERROR', 'Could not get EWN coverage UUID.')], []
        self._ewn_comp.data.info.attrs['cov_uuid'] = self._ewn_comp.cov_uuid
        self._ewn_comp.data.commit()
        # Send the default material list to XMS.
        self._ewn_comp.display_option_list.append(
            XmsDisplayMessage(file=self._ewn_comp.disp_opts_file, edit_uuid=self._ewn_comp.cov_uuid)
        )

    def ensure_display_options_exist(self):
        """Make sure a display options json file exists for the component."""
        if not os.path.exists(self._ewn_comp.disp_opts_file):
            # Read the default display options, and save ourselves a copy with a randomized UUID.
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            categories.target_type = TargetType.polygon
            categories.comp_uuid = self._ewn_comp.uuid
            df = self._ewn_comp.data.meta_data
            classifications = ['Unassigned'] + df['Feature type/Region'].tolist()
            next_color = 0
            for i, cl in enumerate(classifications):
                if next_color >= len(ewn_consts.DEFAULT_MATERIAL_COLORS):
                    next_color = 0
                cat = CategoryDisplayOption()
                cat.description = cl
                cat.file = f'cat_{i}.ids'
                cat.id = i
                cat.options = PolygonOptions()
                clr = ewn_consts.DEFAULT_MATERIAL_COLORS[next_color]
                cat.options.color = QColor(clr[0], clr[1], clr[2])
                cat.options.texture = PolygonTexture.null_pattern
                if i == 0:
                    cat.options.color = QColor(0, 0, 0)
                    cat.is_unassigned_category = True
                    cat.options.texture = PolygonTexture.diagonal_cross_pattern
                categories.categories.append(cat)
                next_color += 1
            write_display_options_to_json(self._ewn_comp.disp_opts_file, categories)
            # Save our display list UUID to the main file
            self._ewn_comp.data.info.attrs['display_uuid'] = categories.uuid
            self._ewn_comp.data.commit()

    def ensure_sediment_display_options_exist(self):
        """Make sure a display options json file exists for the component."""
        if not os.path.exists(self._ewn_comp.disp_opts_file):
            # Read the default display options, and save ourselves a copy with a randomized UUID.
            categories = CategoryDisplayOptionList()  # Generates a random UUID key for the display list
            categories.target_type = TargetType.polygon
            categories.comp_uuid = self._ewn_comp.uuid
            next_color = 0
            for idx, name in ewn_consts.EWN_POLYGON_SEDIMENT_TYPES.items():
                if next_color >= len(ewn_consts.DEFAULT_MATERIAL_COLORS):
                    next_color = 0
                cat = CategoryDisplayOption()
                cat.description = name
                cat.file = f'sediment_{idx}.ids'
                cat.id = idx
                cat.options = PolygonOptions()
                clr = ewn_consts.DEFAULT_MATERIAL_COLORS[next_color]
                cat.options.color = QColor(clr[0], clr[1], clr[2])
                cat.options.texture = PolygonTexture.null_pattern
                if idx == 0:
                    cat.options.color = QColor(0, 0, 0)
                    cat.is_unassigned_category = True
                    cat.options.texture = PolygonTexture.diagonal_cross_pattern
                categories.categories.append(cat)
                next_color += 1
            write_display_options_to_json(self._ewn_comp.disp_opts_file, categories)
            # Save our display list UUID to the main file
            self._ewn_comp.data.info.attrs['display_uuid'] = categories.uuid
            self._ewn_comp.data.commit()

    def assign_polygon_features(self, query, parent, sel_poly_ids):
        """Runs the Assign EWN Features dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            parent (:obj:`QWidget`): Parent window
            sel_poly_ids (:obj:`list`): XMS ids of selected polygons
        """
        dlg_data = {}
        dlg_data['query'] = query
        prj = query.display_projection
        dlg_data['projection'] = prj
        dlg_data['vertical_units'] = prj.vertical_units
        self._sel_poly_ids = sel_poly_ids
        # Don't really need to Query for all polygon component ids here since we don't have sets,
        # but it allows us to cleanup unused component ids when we're done.
        query.load_component_ids(self._ewn_comp, polygons=True)
        self._get_selected_poly_comp_ids()
        comp_id = next(iter(self._sel_poly_comp_ids)) if self._sel_poly_comp_ids else ewn_consts.UNINITIALIZED_COMP_ID
        poly_data = self._ewn_comp.data.get_poly_atts(comp_id)
        dlg_data['poly_data'] = poly_data
        dlg_data['df'] = self._ewn_comp.data.meta_data
        # Warn if polygons with different component ids are selected.
        dlg_data['atts_mismatch'] = len(self._sel_poly_comp_ids) > 1

        cov_geom = query.parent_item()
        # get the selected polygon
        if len(sel_poly_ids) < 2:
            for poly in cov_geom.polygons:
                if poly.id == sel_poly_ids[0]:
                    from xms.ewn.tools.runners import runner_util
                    dlg_data['polygon'] = poly
                    dlg_data['polygon_id'] = poly.id
                    dlg_data['poly_pts'] = runner_util.get_polygon_outside_points_ccw(poly)
                    # find the polygon that contains this polygon
                    for poly1 in cov_geom.polygons:
                        if poly1.id == sel_poly_ids[0]:
                            continue
                        pts = runner_util.get_polygon_outside_points_ccw(poly1)
                        if geoms.point_in_polygon_2d(pts, dlg_data['poly_pts'][0]) == 1:
                            dlg_data['transition_poly_pts'] = pts
        dlg_data['temp_dir'] = Xme.xms_environ_process_temp_directory()
        dlg_data['map_html'] = os.path.join(dlg_data['temp_dir'], 'ewn_map.html')

        dlg2 = AssignEwnPolygonDialog(parent, dlg_data)
        if dlg2.exec() == QDialog.Accepted:
            self._update_poly_component(poly_data)
            if dlg2.new_ugrid is not None:
                add_output_ugrid(query, dlg2.new_ugrid)
            if dlg2.transition_poly_cov is not None:
                add_transition_poly_coverage(query, dlg2.transition_poly_cov)

    def assign_arc_features(self, query, parent, sel_arc_ids):
        """Runs the Assign EWN Arc Features dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            parent (:obj:`QWidget`): Parent window
            sel_arc_ids (:obj:`list`): XMS ids of selected arcs
        """
        dlg_data = {}
        dlg_data['query'] = query
        prj = query.display_projection
        dlg_data['projection'] = prj
        dlg_data['vertical_units'] = prj.vertical_units
        self._sel_arc_ids = sel_arc_ids

        query.load_component_ids(self._ewn_comp, arcs=True)
        self._get_selected_arc_comp_ids()
        comp_id = next(iter(self._sel_arc_comp_ids)) if self._sel_arc_comp_ids else ewn_consts.UNINITIALIZED_COMP_ID
        arc_data = self._ewn_comp.data.get_arc_atts(comp_id)
        dlg_data['arc_data'] = arc_data
        dlg_data['df'] = self._ewn_comp.data.meta_data
        # Warn if arcs with different component ids are selected.
        dlg_data['atts_mismatch'] = len(self._sel_arc_comp_ids) > 1

        cov_geom = query.parent_item()
        dlg_data['cov_geom'] = cov_geom
        dlg_data['ewn_comp'] = self._ewn_comp
        # get the selected arc
        if len(sel_arc_ids) < 2:
            for arc in cov_geom.arcs:
                if arc.id == sel_arc_ids[0]:
                    from xms.ewn.tools.runners import runner_util
                    dlg_data['arc'] = arc
                    dlg_data['arc_id'] = arc.id
                    dlg_data['arc_pts'] = runner_util.get_arc_points(arc)
                    # find the polygon that contains this polygon
                    for poly1 in cov_geom.polygons:
                        pts = runner_util.get_arc_points(poly1)
                        if geoms.point_in_polygon_2d(pts, dlg_data['arc_pts'][0]) == 1:
                            dlg_data['transition_poly_pts'] = pts
        dlg_data['temp_dir'] = Xme.xms_environ_process_temp_directory()
        dlg_data['map_html'] = os.path.join(dlg_data['temp_dir'], 'ewn_map.html')

        dlg2 = AssignEwnArcDialog(parent, dlg_data)
        if dlg2.exec() == QDialog.Accepted:
            self._update_arc_component(arc_data)
            if dlg2.new_ugrid is not None:
                add_output_ugrid(query, dlg2.new_ugrid)
            if dlg2.transition_poly_cov is not None:
                add_transition_poly_coverage(query, dlg2.transition_poly_cov)

    def assign_sediment_properties(self, query, parent, sel_poly_ids):
        """Runs the Sediment Volume Management properties dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            parent (:obj:`QWidget`): Parent window
            sel_poly_ids (:obj:`list`): XMS ids of selected polygons
        """
        self._sel_poly_ids = sel_poly_ids
        # Don't really need to Query for all polygon component ids here since we don't have sets,
        # but it allows us to cleanup unused component ids when we're done.
        query.load_component_ids(self._ewn_comp, polygons=True)
        self._get_selected_poly_comp_ids()
        comp_id = next(iter(self._sel_poly_comp_ids)) if self._sel_poly_comp_ids else ewn_consts.UNINITIALIZED_COMP_ID
        poly_data = self._ewn_comp.data.get_poly_atts(comp_id)

        atts_mismatch = len(self._sel_poly_comp_ids) > 1  # Warn if polygons with different component ids are selected.

        # Compute the number of available priority bins.
        num_available = 0
        if self._ewn_comp.data.polygons.sizes['comp_id'] > 0:
            grouped_dset = self._ewn_comp.data.polygons.groupby('sediment_type')

            for type_idx, dset in grouped_dset:
                if type_idx in [ewn_consts.SEDIMENT_TYPE_AVAILABLE_FILL, ewn_consts.SEDIMENT_TYPE_AVAILABLE_CUT]:
                    num_available += dset.sediment_type.size

        dlg = AssignSedimentVolumeDialog(parent, poly_data, atts_mismatch, num_available)
        if dlg.exec() == QDialog.Accepted:
            self._update_poly_component(poly_data)

    def do_ewn_feature_properties(self, parent):
        """Opens the EWN coverage attributes and display options dialog.

        Args:
            parent (:obj:`PySide2.QtWidgets.QWidget`): The parent window container.
        """
        poly_categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self._ewn_comp.disp_opts_file)
        poly_categories.from_dict(json_dict)

        dlg = EwnPropertiesDialog(parent, poly_categories, self._ewn_comp.data)
        if dlg.exec():  # Update display options json file if dialog accepted
            category_list = dlg.get_category_lists()[0]  # Only one list
            write_display_options_to_json(self._ewn_comp.disp_opts_file, category_list)
            self._ewn_comp.display_option_list.append(
                XmsDisplayMessage(file=self._ewn_comp.disp_opts_file, edit_uuid=self._ewn_comp.cov_uuid)
            )
            self._ewn_comp.data.commit()

    def do_sediment_properties(self, parent):
        """Opens the EWN sediment management display options dialog.

        Args:
            parent (:obj:`PySide2.QtWidgets.QWidget`): The parent window container.
        """
        poly_categories = CategoryDisplayOptionList()
        json_dict = read_display_options_from_json(self._ewn_comp.disp_opts_file)
        poly_categories.from_dict(json_dict)

        dlg = SedimentPropertiesDialog(parent, poly_categories, self._ewn_comp.data)
        if dlg.exec():  # Update display options json file if dialog accepted
            category_list = dlg.get_category_lists()[0]  # Only one list
            write_display_options_to_json(self._ewn_comp.disp_opts_file, category_list)
            self._ewn_comp.display_option_list.append(
                XmsDisplayMessage(file=self._ewn_comp.disp_opts_file, edit_uuid=self._ewn_comp.cov_uuid)
            )
            self._ewn_comp.data.commit()

    def do_dredge_calculator(self, query, parent):
        """Runs the Dredge Calculator dialog.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            parent (:obj:`QWidget`): Qt parent

        Returns:
            (:obj:`list`): Error message tuple in xms.components format if encountered, empty list otherwise
        """
        error = self.get_project_explorer_for_command(query)
        if error:
            return [('ERROR', error)]
        query.load_component_ids(self._ewn_comp, polygons=True)

        # Get the display projection.
        projection = query.display_projection

        dlg = DredgeCalculatorDialog(self._ewn_comp, self._pe_tree, query, projection, parent)
        if dlg.exec() == QDialog.Accepted:
            # Update the polygon attributes and sediment coverage attributes.
            no_dredge_dset = self._ewn_comp.data.polygons.where(
                self._ewn_comp.data.polygons.sediment_type == ewn_consts.SEDIMENT_TYPE_NONE, drop=True
            )
            dredge_dset, cov_attrs = dlg.dialog_data_to_xarray()
            self._ewn_comp.data.polygons = xr.concat([no_dredge_dset, dredge_dset], 'comp_id')
            self._ewn_comp.data.sediment.attrs['target_geom'] = cov_attrs['target_geom']
            self._ewn_comp.data.sediment.attrs['maximum_slope'] = cov_attrs['maximum_slope']
            self._ewn_comp.data.sediment.attrs['output_geom_name'] = cov_attrs['output_geom_name']
            self._ewn_comp.data.sediment.attrs['total_cut'] = cov_attrs['total_cut']
            self._ewn_comp.data.sediment.attrs['total_fill'] = cov_attrs['total_fill']
            self._ewn_comp.data.commit()
            if dlg.sed_comps is not None and dlg.sed_comps.output_geom is not None:
                query.add_ugrid(dlg.sed_comps.output_geom)

        return []

    def get_project_explorer_for_command(self, query):
        """Get the XMS project explorer for a right-click command and verify that a 2D Mesh or UGrid exists.

        Args:
            query (:obj:`xms.api.dmi.Query`): XMS interprocess communication object

        Returns:
            (:obj:`str`): Empty string if 2D Mesh or UGrid exists. Error message if there is no valid geometry
            in the tree
        """
        # Get the current XMS project explorer tree.
        self._pe_tree = query.project_tree
        if self._pe_tree is None:
            return 'Unable to retrieve the SMS project explorer tree.'

        for root_child in self._pe_tree.children:  # Loop through children of the project tree root
            if root_child.children and root_child.name in ['Mesh Data', 'UGrid Data', 'Cartesian Grid Data']:
                # Found either the 2D Mesh root or the UGrid root and it has children. Assume there is at least
                # one valid geometry.
                return ''
        return 'A 2D Mesh or UGrid is required to run this command.'

    def select_insert_features_inputs(self, parent):
        """Displays a dialog for gathering user inputs for the insert features command.

        Args:
            parent (:obj:`PySide2.QtWidgets.QWidget`): The Qt parent window container

        Returns
            (:obj:`dict`): The target geometry UUID, the input EWN Feature coverages' UUIDs
        """
        # Have the user select a target 2D Mesh or UGrid and input EWN Feature coverages.
        dlg = ToolInputDialog(
            self._pe_tree, self._ewn_comp.data.info.attrs['cov_uuid'], ToolInputDialog.INSERT_FEATURES, parent
        )
        if dlg.exec():
            inputs = dlg.tool_inputs()
            return inputs
        return None

    def select_insert_levee_inputs(self, parent, query):
        """Displays a dialog for gathering user inputs for the insert features command.

        Args:
            parent (:obj:`PySide2.QtWidgets.QWidget`): The Qt parent window container
            query (:obj:`xms.api.dmi.Query`): XMS interprocess communication object

        Returns
            (:obj:`dict`): The target geometry UUID, the input EWN Feature coverages' UUIDs
        """
        dlg = InsertLeveeDialog(parent, self._pe_tree)
        if dlg.exec():
            inputs = dlg.tool_inputs()
            return inputs
        return None

    def select_generate_roughness_inputs(self, parent, query):
        """Displays a dialog for gathering user inputs for the generate roughness dataset command.

        Args:
            parent (:obj:`PySide2.QtWidgets.QWidget`): The Qt parent window container
            query (:obj:`xms.api.dmi.Query`): XMS interprocess communication object

        Returns
            (:obj:`dict`): The user input. See ToolInputDialog for keys.
        """
        # Have the user select a target 2D Mesh or UGrid, input EWN Feature coverages, and an initial roughness.
        dlg = ToolInputDialog(
            self._pe_tree, self._ewn_comp.data.info.attrs['cov_uuid'], ToolInputDialog.ROUGHNESS_DATASET, parent, query
        )
        if dlg.exec():
            return dlg.tool_inputs()
        return None

    def select_update_adh_friction_inputs(self, parent, query):
        """Displays a dialog for gathering user inputs for the updating adh friction inputs.

        Args:
            parent (:obj:`PySide2.QtWidgets.QWidget`): The Qt parent window container
            query (:obj:`xms.api.dmi.Query`): XMS interprocess communication object

        Returns
            (:obj:`dict`): The user input. See UpdateAdhFrictionDialog for keys.
        """
        # Have the user select a target adh materials coverage.
        dlg = UpdateAdhFrictionDialog(parent, self._pe_tree, self._ewn_comp.cov_uuid)
        if dlg.exec():
            return dlg.tool_inputs()
        return None

    def select_refine_quadtree_inputs(self, parent, query):
        """Select refinment inputs."""
        dlg = RefineQuadTreeDialog(parent, self._pe_tree, self._ewn_comp.cov_uuid)
        if dlg.exec():
            return dlg.tool_inputs()
        return None
