"""Xarray data class for the EWN Sediment Management coverage component."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
import pkg_resources
import xarray as xr

# 3. Aquaveo modules
from xms.components.bases.xarray_base import XarrayBase
from xms.core.filesystem import filesystem as xmf

# 4. Local modules
from xms.ewn.data import ewn_cov_data_consts as ewn_consts


class SedimentCovData(XarrayBase):
    """Class for storing the EWN Sediment Management  coverage properties."""
    def __init__(self, filename):
        """Construct the data class."""
        super().__init__(filename)
        self._polygons = None  # polygon data set
        self._sediment = None
        self._cur_version = pkg_resources.get_distribution('xmsewn').version
        self._ensure_info_exists()

    def _ensure_info_exists(self):
        """Make sure all the info Dataset attrs are initialized."""
        self.info.attrs['FILE_TYPE'] = 'EWN_SEDIMENT_MANAGEMENT_DATA'
        if 'cov_uuid' not in self.info.attrs:
            self.info.attrs['cov_uuid'] = ''  # gets set later
        if 'display_uuid' not in self.info.attrs:
            self.info.attrs['display_uuid'] = ''
        if 'VERSION' not in self.info.attrs:
            self.info.attrs['VERSION'] = self._cur_version

    @property
    def polygons(self):
        """Get the polygons data parameters.

        Returns:
            (:obj:`xarray.Dataset`): The polygons dataset

        """
        if self._polygons is None:
            self._polygons = self.get_dataset('polygons', False)
            if self._polygons is None:
                self._polygons = self._new_polygon_dataset(False)
        return self._polygons

    @polygons.setter
    def polygons(self, dset):
        """Set the polygons data parameters.

        Args:
            dset (:obj:`xarray.Dataset`): The new polygons Dataset

        """
        if dset is not None:
            self._polygons = dset

    @property
    def sediment(self):
        """Get the sediment data parameters.

        Returns:
            (:obj:`xarray.Dataset`): The sediment dataset

        """
        if self._sediment is None:
            self._sediment = self.get_dataset('sediment', False)
            if self._sediment is None:
                self._sediment = self._new_sediment_dataset()
        return self._sediment

    @sediment.setter
    def sediment(self, dset):
        """Set the sediment data parameters.

        Args:
            dset (:obj:`xarray.Dataset`): The new sediment Dataset

        """
        if dset is not None:
            self._sediment = dset

    def _new_polygon_dataset(self, default_values):
        """Creates a new polygon feature Dataset.

        Args:
            default_values (:obj:`bool`): If True, populate one entry in the dataset with default values

        Returns:
            (:obj:`xarray.Dataset`): An empty polygon feature dataset
        """
        # Construct the Dataset
        if default_values:  # Default dataset for one element
            comp_id = 0 if self.polygons.comp_id.size == 0 else max(self.polygons.comp_id).item() + 1
            coords = {'comp_id': np.array([comp_id], dtype=np.int32)}
            poly_table = {
                'polygon_name': ('comp_id', np.array([''], dtype=np.unicode_)),
                # Dredging attributes
                'sediment_type': ('comp_id', np.array([ewn_consts.SEDIMENT_TYPE_NONE], dtype=np.int32)),
                'priority': ('comp_id', np.array([1], dtype=np.int32)),
                'priority_percent': ('comp_id', np.array([100.0], dtype=np.float64)),
                'cut_fill_type': ('comp_id', np.array([ewn_consts.CUT_FILL_TYPE_CONSTANT], dtype=np.int32)),
                'volume_value': ('comp_id', np.array([0.0], dtype=np.float64)),
                # Computed dredging values
                'total_volume': ('comp_id', np.array([0.0], dtype=np.float64)),
                'required_volume': ('comp_id', np.array([0.0], dtype=np.float64)),
                'available_volume': ('comp_id', np.array([0.0], dtype=np.float64)),
                'cut_volume': ('comp_id', np.array([0.0], dtype=np.float64)),
                'fill_volume': ('comp_id', np.array([0.0], dtype=np.float64)),
            }
        else:  # Empty dataset
            coords = {'comp_id': np.array([], dtype=np.int32)}
            poly_table = {
                'polygon_name': ('comp_id', np.array([], dtype=np.unicode_)),
                # Dredging attributes
                'sediment_type': ('comp_id', np.array([], dtype=np.int32)),
                'priority': ('comp_id', np.array([], dtype=np.int32)),
                'priority_percent': ('comp_id', np.array([], dtype=np.float64)),
                'cut_fill_type': ('comp_id', np.array([], dtype=np.int32)),
                'volume_value': ('comp_id', np.array([], dtype=np.float64)),
                # Computed dredging values
                'total_volume': ('comp_id', np.array([], dtype=np.float64)),
                'required_volume': ('comp_id', np.array([], dtype=np.float64)),
                'available_volume': ('comp_id', np.array([], dtype=np.float64)),
                'cut_volume': ('comp_id', np.array([], dtype=np.float64)),
                'fill_volume': ('comp_id', np.array([], dtype=np.float64)),
            }
        return xr.Dataset(data_vars=poly_table, coords=coords)

    @staticmethod
    def _new_sediment_dataset():
        """Creates a new sediment properties Dataset.

        Returns:
            (:obj:`xarray.Dataset`): An empty polygon feature dataset
        """
        # Construct the Dataset
        attrs = {
            'target_geom': '',  # Target geometry
            'maximum_slope': 0.0,  # Maximum Slope
            'output_geom_name': '',  # Output Geometry Name
            'total_cut': 0.0,  # Total Cut
            'total_fill': 0.0  # Total Fill
        }
        return xr.Dataset(attrs=attrs)

    def get_poly_atts(self, comp_id):
        """Gets a record from a component id. If the id is not known then returns None.

        Args:
            comp_id (:obj:`int`): The polygon's component id

        Returns:
            (:obj:`xarray.Dataset`): The record from the polygons dataset
        """
        if comp_id == ewn_consts.UNINITIALIZED_COMP_ID:
            return self._new_polygon_dataset(True)  # Polygon's component id has not been set yet, return default data.

        poly_data = self.polygons.where(self.polygons.comp_id == comp_id, drop=True)
        if len(poly_data.comp_id) == 0:  # Polygon's component id not found, return default data.
            return self._new_polygon_dataset(True)  # Return default data

        # Polygon's component id found in Dataset, populate the param object from existing attributes.
        return poly_data

    def add_polygon(self, poly_data):
        """Append the polygon feature attributes for a new polygon to the dataset.

        Args:
            poly_data (:obj:`xarray.Dataset`): Polygon attribute dataset

        Returns:
            (:obj:`tuple(int)`): The newly generated component id
        """
        try:
            if len(self.polygons.comp_id) == 0:  # Empty dataset
                new_comp_id = 0
            else:  # Generate a new, unique component id
                new_comp_id = int(max(self.polygons.comp_id.data)) + 1
            # Reset the comp_id coord for this data so we can append it to our dataset.
            poly_data = poly_data.assign_coords({'comp_id': np.array([new_comp_id], dtype=np.int32)})
            self._polygons = xr.concat([self.polygons, poly_data], 'comp_id')
            return new_comp_id
        except Exception:  # pragma no cover
            return ewn_consts.UNINITIALIZED_COMP_ID

    def remove_poly_comp_ids(self, delete_comp_ids):
        """Removes comp ids from the class that are not used any more.

        Args:
            delete_comp_ids (:obj:`iterable`): List-like containing the component ids to delete

        """
        if not delete_comp_ids:
            return
        # Drop unused component ids from the datasets
        mask = self.polygons.comp_id.isin(list(delete_comp_ids))
        self.polygons = self.polygons.where(~mask, drop=True)

    def close(self):
        """Closes the H5 file and does not write any data that is in memory."""
        super().close()
        if self._polygons is not None:
            self._polygons.close()
        if self._sediment is not None:
            self._sediment.close()

    def commit(self):
        """Save in memory datasets to the NetCDF file."""
        # Store the current package version.
        self.info.attrs['VERSION'] = self._cur_version
        self.close()
        super().commit()
        if self._polygons is not None:  # Only write polygon Dataset to disk if it has been loaded into memory.
            self._polygons.close()
            self._drop_h5_groups(['polygons'])
            self._polygons.to_netcdf(self._filename, group='polygons', mode='a')
        if self._sediment is not None:  # Only write sediment Dataset to disk if it has been loaded into memory.
            self._sediment.close()
            self._drop_h5_groups(['sediment'])
            self._sediment.to_netcdf(self._filename, group='sediment', mode='a')

    def vacuum(self):
        """Rewrite all SimData to a new/wiped file to reclaim disk space.

        All BC datasets that need to be written to the file must be loaded into memory before calling this method.
        """
        _ = self.info  # Ensure all data is loaded into memory.
        _ = self.polygons
        _ = self.sediment
        xmf.removefile(self._filename)  # Delete the existing NetCDF file.
        self.commit()  # Rewrite all datasets.
