"""Dialog for assigning Sediment coverage properties."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtWidgets import QAbstractItemView, QDialogButtonBox, QGroupBox, QHBoxLayout, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.widgets.category_display_options_list import CategoryDisplayOptionsWidget, Columns

# 4. Local modules


class SedimentPropertiesDialog(XmsDlg):
    """A dialog to define EWN feature properties and their display options."""
    def __init__(self, parent, display_options, data):
        """Constructor.

        Args:
            parent (:obj:`QWidget`): Parent Qt dialog
            display_options (:obj:`CategoryDisplayOptionList`): The current EWN display options
            data (:obj:`EwnCovData`): The coverage component data
        """
        super().__init__(parent, 'xms.ewn.gui.sediment_properties_dialog')
        self._help_url = 'https://www.xmswiki.com/wiki/SMS:Display_Options'
        self._data = data
        self._widgets = {}
        self._setup_ui(display_options)

    def _setup_ui(self, display_options):
        """Setup dialog widgets.

        Args:
            display_options (:obj:`CategoryDisplayOptionList`): The current EWN display options
        """
        self.setWindowTitle('Sediment Management Properties')
        self._setup_display_options(display_options)
        self._setup_main_layout()

    def _setup_display_options(self, display_options):
        """Adds the display options table view.

        Args:
            display_options (:obj:`CategoryDisplayOptionList`): The current EWN display options
        """
        # Setup the display options widget
        self._widgets['display_options'] = CategoryDisplayOptionsWidget(self)
        display_widget = self._widgets['display_options']
        display_widget.set_category_lists([display_options])
        display_widget.ui.table_view_categories.setColumnHidden(Columns.FONT_CHECK, True)
        display_widget.ui.table_view_categories.setColumnHidden(Columns.FONT, True)
        display_widget.ui.btn_change_all_label.setVisible(False)
        self._widgets['grp_display_options'] = QGroupBox('Display options')
        self._widgets['layout_display_options'] = QVBoxLayout()
        self._widgets['layout_display_options'].addWidget(display_widget)
        self._widgets['grp_display_options'].setLayout(self._widgets['layout_display_options'])

        display_widget.ui.table_view_categories.setSelectionBehavior(QAbstractItemView.SelectRows)
        display_widget.ui.table_view_categories.setSelectionMode(QAbstractItemView.SingleSelection)

    def _setup_main_layout(self):
        """Adds the dialog button bar and sets the top-level dialog layout."""
        # Add the dialog buttons layout
        self._widgets['btn_box'] = QDialogButtonBox()
        btn_flags = QDialogButtonBox.Ok | QDialogButtonBox.Cancel | QDialogButtonBox.Help
        self._widgets['btn_box'].setStandardButtons(btn_flags)
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)
        self._widgets['btn_horiz_layout'] = QHBoxLayout()
        self._widgets['btn_horiz_layout'].addWidget(self._widgets['btn_box'])

        # Add a main layout.
        self._widgets['main_horiz_layout'] = QHBoxLayout()
        self._widgets['main_horiz_layout'].addWidget(self._widgets['grp_display_options'])
        self._widgets['main_vert_layout'] = QVBoxLayout()
        self._widgets['main_vert_layout'].addLayout(self._widgets['main_horiz_layout'])
        self._widgets['main_vert_layout'].addLayout(self._widgets['btn_horiz_layout'])
        self.setLayout(self._widgets['main_vert_layout'])

    def help_requested(self):  # pragma: no cover
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def accept(self):
        """Ensure currently selected feature type's attributes get saved."""
        super().accept()

    def get_category_lists(self):
        """Returns the current category display options lists.

        Returns:
            (:obj:`list[CategoryDisplayOptionList]`): The current category display options list.
        """
        return self._widgets['display_options'].get_category_lists()
