"""StationComponent class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.gmi.component_bases.coverage_component_base import CoverageComponentBase as GmiCoverageComponent
from xms.gmi.data.generic_model import Section
from xms.guipy.data.plot_and_table_data_srh import PlotsAndTableDataSrh
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.plot_and_table_dialog import PlotsAndTableDialog


# 4. Local modules
from xms.funwave.data.model import get_model


class StationComponent(GmiCoverageComponent):
    """A Dynamic Model Interface (DMI) component base for the FUNWAVE output stations coverage."""
    def __init__(self, main_file: Optional[str | Path]):
        """
        Initialize the component.

        Args:
            main_file: The component's main-file.
        """
        super().__init__(main_file)

        self.point_commands = [('Display point plots', self.show_plot), ]

    def _get_section(self, target: TargetType) -> Section:
        """Get a Section with parameters for a given target."""
        return get_model().section_from_target_type(target)

    def show_plot(self, query: Query, _params, parent: QWidget):
        """Open the Observation dialog and save component data state on OK.

        Args:
            query (Query): Query object
            _params (dict): Parameters passed to the function as a dictionary
            parent (QWidget): The QWidget parent
        """
        plots_and_table = PlotsAndTableDataSrh(query.project_tree, _params[0]['selection'][0], 'Point', self.cov_uuid,
                                               'FUNWAVE', feature_ids=_params[0]['selection'])
        dlg = PlotsAndTableDialog(parent, data=plots_and_table)
        dlg.exec()
        return [], []
