"""SimData class."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Section
from xms.gmi.data.generic_model import Type
from xms.gmi.data_bases.sim_base_data import SimBaseData

# 4. Local modules
from xms.funwave.data.model import _value_to_dict, get_model
from xms.funwave.data.wavemaker import WAVE_MAKER_OPTIONS

SIM_DATA_MAINFILE = 'sim_comp.nc'


class SimData(SimBaseData):
    """Manages data for the simulation component."""

    @property
    def model_control_data(self) -> Section:
        """Generic model section with the model control parameters."""
        section = get_model().global_parameters
        vals = self.global_values
        if vals:
            section.restore_values(vals)
        return section

    def set_values_in_model_control(self, values: dict) -> None:
        """Set the values in the model control section.

        Args:
            values (:obj:`dict`): The values to set.
        """
        _global_values = get_model().global_parameters
        group_names = self.model_control_data.group_names
        for group_name in group_names:
            group = self.model_control_data.group(group_name)
            parameter_names = group.parameter_names
            for parameter_name in parameter_names:
                parameter = group.parameter(parameter_name)
                name = parameter.parameter_name
                if name in values:
                    value = values[name]
                    if name == 'VISCOSITY_BREAKING':
                        if values[name]:
                            value = 'Eddy-Viscosity breaking used in numerical scheme (VISCOSITY_BREAKING=T & '\
                                'SHOW_BREAKING=T)'
                        else:
                            value = 'Shock-Capturing breaking used in numerical scheme (VISCOSITY_BREAKING=F & '\
                                'SHOW_BREAKING=F)'
                    if name in ['SHOW_BREAKING', 'COMPLEXITY', 'unimplemented']:
                        continue
                    if name == 'WAVEMAKER':
                        if values[name] in WAVE_MAKER_OPTIONS:
                            value = WAVE_MAKER_OPTIONS[values[name]]
                    if name == 'U':
                        values['Velocity'] = value
                        values.pop('U', None)
                    if name == 'V':
                        values['Velocity'] = value
                        values.pop('V', None)
                    if name == 'Umean':
                        values['Average velocity'] = value
                        values.pop('Umean', None)
                    if name == 'Vmean':
                        values['Average velocity'] = value
                        values.pop('Vmean', None)

                    # parameter.value = value
                    _global_values.group(group_name).parameter(parameter_name).value = value
        self.global_values = _global_values.extract_values()

    def get_dataset_uuids(self) -> dict[str, str]:
        """Get the datasets that need to be written.

        Returns:
            (:obj:`dict`): Dictionary of dataset name to DatasetReader.
        """
        datasets = {}
        group_names = self.model_control_data.group_names
        for group_name in group_names:
            group = self.model_control_data.group(group_name)
            parameter_names = group.parameter_names
            for parameter_name in parameter_names:
                parameter = group.parameter(parameter_name)
                if parameter.parameter_type == Type.DATASET:
                    datasets[parameter_name] = parameter.value
        return datasets

    def values_to_dict(self) -> dict:
        """Convert the data values to a dictionary.

        Returns:
            The dictionary of values.
        """
        values = {}
        group_names = self.model_control_data.group_names
        for group_name in group_names:
            group = self.model_control_data.group(group_name)
        # sections = self.model_control_data
        # for group in sections.groups:
            parameter_names = group.parameter_names
            for parameter_name in parameter_names:
                parameter = group.parameter(parameter_name)
                name = parameter.parameter_name

                values = _value_to_dict(values, parameter, name)

        values['RESULT_FOLDER'] = './output/'
        return values
