"""Class to read a FUNWAVE control input file."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import os
from pathlib import Path
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.constraint import Numbering, Orientation, RectilinearGridBuilder, UGrid2d
from xms.data_objects.parameters import Component, Coverage, Point, Simulation, UGrid as DoUGrid

# 4. Local modules
from xms.funwave.components.sim_component import get_component_data_object
from xms.funwave.components.station_component import StationComponent
import xms.funwave.data.sim_data as smd
from xms.funwave.file_io.dataset_reader import build_datasets
from xms.funwave.file_io.input_txt_reader import read_input_txt_file
import xms.funwave.file_io.io_util as io_util


class ControlReader:
    """Class to read a FUNWAVE input file."""

    def __init__(self, filename: str):
        """Constructor.

        Args:
            filename: The input file.
        """
        self._logger = logging.getLogger('xms.funwave')
        self._filename = Path(os.path.normpath(filename))

        # Outputs
        self._do_sim = None
        self._do_sim_comp = None
        self._do_ugrid = None
        self._datasets = []
        self._coverages = []

        self._sim_name = None
        self._sim_uuid = None
        self._sim_data = None
        self._sim_comp_uuid = None

    def _reset(self, filename):
        """Reset member variables before reading a control file.

        Args:
            filename (:obj:`str`): Absolute path to the control file we are about to read
        """
        self._sim_name = os.path.splitext(os.path.basename(filename))[0]
        self._sim_uuid = str(uuid.uuid4())
        self._sim_data = None
        self._sim_comp_uuid = str(uuid.uuid4())

    def _build_xms_data(self):
        """Add all the imported data to the xmsapi Query to send back to SMS."""
        # Add the simulation
        self._do_sim = Simulation(model='FUNWAVE', sim_uuid=self._sim_uuid, name=self._sim_name)
        self._do_sim_comp = get_component_data_object(
            main_file=self._sim_data._filename, comp_uuid=self._sim_comp_uuid, unique_name='SimComponent'
        )

        # Set the projection for all the coverages read with this simulation
        # for bc_cov, _ in self._bc_coverages[-1]:
        #     bc_cov.projection = self._do_projection

    def _add_build_data(self, query):
        """Adds build data at the end of reading everything.

        Args:
            query (:obj:`Query`): The XMS interprocess communicator
        """
        # Add the simulations
        query.add_simulation(self._do_sim, components=[self._do_sim_comp])

        # Add the geometric data read from the file
        query.add_ugrid(self._do_ugrid)
        query.link_item(self._sim_uuid, self._do_ugrid.uuid)
        for dset in self._datasets:
            query.add_dataset(dset)

        # Add the point station output coverages
        comp = None
        for cov in self._coverages:
            comp = create_station_output_component(cov.uuid)  # , query)
            query.add_coverage(cov, model_name='FUNWAVE', coverage_type='Output Stations', components=[comp])
            query.link_item(self._sim_uuid, cov.uuid)

    def _read_sim(self):
        """Read an input.txt file."""
        try:
            self._logger.info(f'Parsing input.txt file: {io_util.logging_filename(self._filename)}')

            input_values, depth_file = read_input_txt_file(Path(self._filename))
            dx = input_values['DX']
            dy = input_values['DY']

            # Build the station output coverage
            number_of_stations = input_values.get('NumberStations', 0)
            if number_of_stations > 0:
                self._coverages = [build_coverages(input_values.get('STATIONS_FILE', ''), dx, dy)]

            # Build the grid
            m_size = input_values['Mglob']
            n_size = input_values['Nglob']
            co_grid = build_co_grid(m_size, n_size, dx, dy)
            ugrid_uuid = str(uuid.uuid4())

            # Build datasets
            datasets = build_datasets(input_values, self._filename, ugrid_uuid, depth_file)
            self._datasets = []
            for name, dataset_writer, dataset_uuid, values in datasets:
                if name == 'DEPTH_FILE':
                    values = -values
                    co_grid.cell_elevations = values
                else:
                    input_values[name] = dataset_uuid
                    self._datasets.append(dataset_writer)

            # Now build the Do_Ugrid
            self._do_ugrid = build_grid(co_grid, ugrid_uuid)

            # Create the simulation data
            self._sim_data = smd.SimData(
                os.path.join(io_util.create_component_folder(self._sim_comp_uuid), smd.SIM_DATA_MAINFILE)
            )
            self._sim_data.set_values_in_model_control(input_values)

            self._sim_data.info.attrs['domain_uuid'] = self._do_ugrid.uuid

            self._logger.info('Committing changes....')
            self._sim_data.commit()

            if 'TITLE' in input_values and input_values['TITLE'] != '':
                self._sim_name = input_values['TITLE']
            self._build_xms_data()
            self._logger.info('Finished!')
        except Exception:
            self._logger.exception('Unexpected error reading input.funwave file.')

    def send(self, query):
        """Send all the imported data back to SMS.

        Args:
            query (:obj:`Query`): The XMS interprocess communicator
        """
        if query:
            self._add_build_data(query)
            query.send()

    def read(self):
        """Read all the FUNWAVE simulations."""
        if not os.path.isfile(self._filename):
            self._logger.error(f'Unable to find input file: {self._filename}')
        else:
            self._reset(str(self._filename))
            self._read_sim()


def build_grid(grid: UGrid2d, ugrid_uuid: str) -> DoUGrid:
    """Build data objects grid from input.funwave values.

    Args:
        m_size(:obj:`int`): The number of cells along the x-axis.
        n_size(:obj:`int`): The number of cells along the y-axis.
        dx(:obj:`float`): The cell size along the x-axis.
        dy(:obj:`float`): The cell size along the y-axis.
        ugrid_uuid (str): UUID of the UGrid

    Returns:
        The grid.
    """
    xmc_file = os.path.join(XmEnv.xms_environ_process_temp_directory(), f'{ugrid_uuid}.xmc')
    grid.write_to_file(xmc_file, True)
    do_ugrid = DoUGrid(xmc_file, name='Grid', uuid=ugrid_uuid)
    return do_ugrid


def build_co_grid(m_size: int, n_size: int, dx: float, dy: float) -> UGrid2d:
    """Build constrained grid from input.funwave values.

    Args:
        m_size(:obj:`int`): The number of cells along the x-axis.
        n_size(:obj:`int`): The number of cells along the y-axis.
        dx(:obj:`float`): The cell size along the x-axis.
        dy(:obj:`float`): The cell size along the y-axis.

    Returns:
        The grid.
    """
    builder = RectilinearGridBuilder()
    builder.locations_x = [m * dx for m in range(m_size + 1)]
    builder.locations_y = [n * dy for n in range(n_size + 1)]
    builder.origin = (0.0, 0.0, 0.0)
    builder.angle = 0.0
    builder.numbering = Numbering.kji
    builder.orientation = (Orientation.x_increase, Orientation.y_increase)
    grid = builder.build_grid()
    return grid


def build_coverages(station_file, dx, dy):
    """Build a coverage from station_file values.

    Args:
        station_file (:obj:`string`): the location of the x,y locations of points for the station outputs.
        dx (:obj:`float`): The grid size in the x direction
        dy (:obj:`float`): The grid size in the y direction

    Returns:
        The coverage.
    """
    station_points = []
    # Read the Station file
    if station_file == '':
        return None
    station_points = []
    id = 1
    with open(station_file) as f:
        lines = f.readlines()
        for line in lines:
            tokens = line.split()
            if len(tokens) > 1:
                try:
                    i_value = float(tokens[0])
                    j_value = float(tokens[1])
                    x_value = float(i_value - 1) * dx + dx / 2
                    y_value = float(j_value - 1) * dy + dy / 2
                    station_points.append(Point(x=x_value, y=y_value, z=0.0, feature_id=id))
                    id += 1
                except ValueError:
                    pass

    # Check the points
    if len(station_points) <= 0:
        return None

    # Create the coverage
    cov_name = station_file[0: station_file.find('.')]
    cov_uuid = str(uuid.uuid4())
    cov = Coverage(name=cov_name, uuid=cov_uuid)
    cov.set_points(station_points)
    cov.complete()

    return cov


def create_component_mainfile(mainfile):
    """Creates a directory for a mainfile for a component in a new UUID named folder in the temp area.

    Args:
        mainfile (:obj:`str`): The default filename for the mainfile.

    Returns:
        (:obj:`tuple`) A tuple consisting of the new component UUID, the absolute path for the new mainfile, and the new
        component UUID directory.
    """
    comp_uuid = str(uuid.uuid4())
    comp_dir = os.path.join(XmEnv.xms_environ_temp_directory(), 'Components', comp_uuid)
    os.makedirs(comp_dir, exist_ok=True)
    new_mainfile = os.path.join(comp_dir, mainfile)
    return comp_uuid, new_mainfile, comp_dir


def create_station_output_component(cov_uuid):  # , query):
    """Create a component.

    Args:
            cov_uuid (:obj:`str`): The UUID of the coverage
            query (:obj:`Query`): The XMS interprocess communicator
    """
    comp_uuid, station_output_mainfile, _ = create_component_mainfile('station_comp.nc')
    do_comp = Component(main_file=station_output_mainfile, unique_name='StationComponent', comp_uuid=comp_uuid,
                        model_name='FUNWAVE')
    comp = StationComponent(station_output_mainfile)
    comp.data.info.attrs['cov_uuid'] = cov_uuid
    comp.data.info.attrs['model_name'] = 'FUNWAVE'
    comp.data.info.attrs['unique_name'] = 'StationComponent'
    comp.cov_uuid = cov_uuid
    comp.data.commit()
    return do_comp
