"""Utility methods used by the GUI modules."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.tree import tree_util
from xms.core.filesystem import filesystem as io_util
from xms.guipy import settings
from xms.guipy.dialogs.file_selector_dialogs import get_open_filename, get_open_foldername

# 4. Local modules


NULL_SELECTION = '(none selected)'


def get_file_selector_start_dir(label_text, proj_dir):
    """Get the directory to open file browser in.

    Args:
        label_text (:obj:`str`): The GUI label text associated with a file selector
        proj_dir (:obj:`str`): Directory of the saved project if it exists. If the project has been saved, any files
            selected at the time were converted to relative paths from the project directory.

    Returns:
        (:obj:`str`): Directory to open file browser in. If previously selected file, use that folder. Otherwise, use
        the last directory that was stored in the registry.
    """
    start_dir = proj_dir  # Default to the project directory if there is one.
    if label_text != NULL_SELECTION:  # Start in the directory of the last selected file, if there is one.
        if not os.path.isabs(label_text):  # Stored relative to the project directory.
            label_text = io_util.resolve_relative_path(proj_dir, label_text)
        start_dir = os.path.dirname(label_text)
    if not start_dir:  # If no project location and no file previously selected, look in registry
        start_dir = settings.get_file_browser_directory()
    return start_dir


def select_file(parent, label, caption, file_filter, proj_dir, is_directory):
    """Display a file selector dialog.

    Args:
        parent (:obj:`QWidget`): The parent dialog
        label (:obj:`QLabel`): The label widget associated with the file selector.
        caption (:obj:`str`): The dialog caption
        file_filter (:obj:`str`): File extension filter
        proj_dir (:obj:`str`): Directory of the saved project if it exists. If the project has been saved, any files
            selected at the time were converted to relative paths from the project directory.
        is_directory (:obj:`bool`): True if selecting a directory, False for files
    """
    start_dir = get_file_selector_start_dir(label.text(), proj_dir)
    if is_directory:
        filename = get_open_foldername(parent, caption, start_dir)
    else:
        filename = get_open_filename(parent, caption, file_filter, start_dir)
    if filename and os.path.exists(filename):
        label.setText(filename)


def set_combobox_from_data(cbx, value):
    """Set a combobox option index from user data value.

    Args:
        cbx (:obj:`QCombobox`): The combobox to search
        value (:obj:`object`): The user data value

    Returns:
        (:obj:`int`): Index of the data's combobox option or -1 if not found
    """
    index = -1
    try:
        index = cbx.findData(value)
        cbx.setCurrentIndex(max(index, 0))
    except AttributeError:
        pass
    return index


def add_combobox_options(option_map, cbx):
    """Add a combobox's option texts and user data values.

    Args:
        option_map (:obj:`dict`): Mapping of the display text to FUNWAVE data value
        cbx (:obj:`QCombobox`): The combobox widget.
    """
    cbx_name = cbx.objectName()
    model_vals = option_map[cbx_name][1]
    for idx, opt in enumerate(option_map[cbx_name][0]):  # Loop through the list of display option texts
        cbx.addItem(opt, model_vals[idx])  # Add the FUNWAVE value as UserData


def set_widget_text_to_tree_path(tree_node, item_uuid, widget):
    """Set the text of a QWidget to be the tree path of a selected item.

    Args:
        tree_node (:obj:`TreeNode`): The project explorer tree
        item_uuid (:obj:`str`): UUID of the selected item
        widget (:obj:`QWidget`): The QWidget whose text attribute will be updated
    """
    item_name = tree_util.build_tree_path(tree_node, item_uuid)
    if not item_name:
        widget.setText(NULL_SELECTION)
        return

    name_split = item_name.split('/', 2)
    if len(name_split) > 2:  # Chop off project and module roots
        item_name = name_split[2]
    widget.setText(item_name)
