"""Tests utilities."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import math
import os
from sys import platform

# 2. Third party modules
import osgeo.gdal as gdal

# 3. Aquaveo modules

# 4. Local modules


def _is_close(list1, list2, abs_tol):
    if len(list1) != len(list2):
        return False
    for v1, v2 in zip(list1, list2):
        if not math.isclose(v1, v2, abs_tol=abs_tol):
            return False
    return True


def compare_with_gdalinfo(filename1, filename2):
    """Uses gdal.info to compare two raster files and returns the number of differences and the difference strings.

    Args:
        filename1 (str): Path to first file.
        filename2 (str): Path to second file.

    Returns:
        The number of differences and the difference strings.
    """
    # Remove the aux.xml files that may contain custom information. We don't want to compare that information.
    aux1 = filename1 + '.aux.xml'
    if os.path.exists(aux1):
        os.remove(aux1)
    aux2 = filename2 + '.aux.xml'
    if os.path.exists(aux2):
        os.remove(aux2)

    # Run gdalinfo and compare the results between the files.
    gdal.UseExceptions()
    d2 = None
    try:
        d1 = gdal.Info(filename1, format='json', wktFormat='WKT1', computeMinMax=True)
    except RuntimeError:
        # If we get a runtime error, don't compute the min/max values
        d1 = gdal.Info(filename1, format='json', wktFormat='WKT1')
        d2 = gdal.Info(filename2, format='json', wktFormat='WKT1')
    if d2 is None:
        try:
            d2 = gdal.Info(filename2, format='json', wktFormat='WKT1', computeMinMax=True)
        except RuntimeError:
            # If we get a runtime error, don't compute the min/max values
            d2 = gdal.Info(filename2, format='json', wktFormat='WKT1')
            d1 = gdal.Info(filename1, format='json', wktFormat='WKT1')
    info = [(k, d2[k], v) for k, v in d1.items()]
    count = 0
    diff = ''
    tol = 1.0e-7
    for item in info:
        if item[0] != 'files' and item[0] != 'description':
            different = False
            if item[0] == 'geoTransform':
                if not _is_close(item[1], item[2], tol):
                    different = True
            elif item[1] != item[2]:
                different = True
            if different:
                if count != 0:
                    diff += ', '
                count += 1
                diff += item[0] + ': ' + str(item[1]) + ' != ' + str(item[2])
    ret_diff = 'Differences Found: ' + str(count)
    if diff:
        ret_diff += ' Details: ' + diff
    return count, ret_diff


def compare_tif_files(filename1, filename2):
    """Check if tif files exist in the directories and, if so, compare them.

    Args:
        filename1 (str): First file path.
        filename2 (str): Second file path.

    Returns:
        An empty string if files not found or they match, otherwise an error message.
    """
    isfile1 = os.path.isfile(filename1)
    isfile2 = os.path.isfile(filename2)
    error = ''
    if not isfile1:
        error += f'FAIL - missing file: {filename1}\n'
    if not isfile2:
        error += f'FAIL - missing file: {filename2}\n'
    if isfile1 and isfile2:
        return compare_with_gdalinfo(filename1, filename2)
    return -1, error


def linux_not_supported():
    """Check if this is a Windows-only feature that's not supported."""
    return platform != "win32"
