"""Worker thread for the 1D grid mapping operation."""

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackThread

# 4. Local modules
from xms.gencade.dmi.sim_queries import SimQueries
from xms.gencade.mapping.grid_mapper import map_grid


class GridMappingWorkerThread(ProcessFeedbackThread):
    """Worker thread class to do computations in another thread."""

    def __init__(self, query, sim_comp, parent):
        """Constuct the worker.

        Args:
            query (:obj:`Query`): XMS interprocess communication object
            sim_comp (:obj:`SimComponent`): The simulation data
            parent (:obj:`QWidget`): The Qt parent
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self.logger = logging.getLogger('xms.gencade')
        self.query = query
        self.sim_comp = sim_comp
        self.grid_data = {}

    def _add_mapped_data(self, do_comp, py_comp):
        """Send the new mapped 1D grid to XMS, unlink the source coverage.

        Context must be at the simulation level

        Args:
            do_comp (:obj:`Component`): The new mapped grid data_objects component
            py_comp (:obj:`MappedGridComponent`): The new mapped grid Python component
        """
        self.logger.info('Preparing data for loading into SMS.')
        # Add the mapped component to the Context and send back to XMS.
        self.query.add_component(do_comp, display_options=py_comp.get_display_options())
        self.query.link_item(self.grid_data['sim_uuid'], do_comp.uuid)
        # Unlink the source 1D coverage
        self.query.unlink_item(self.grid_data['sim_uuid'], self.grid_data['grid_cov'].uuid)
        # Unlink any old applied grid components.
        for mapped_grid in self.grid_data['mapped_grids']:
            self.query.delete_item(mapped_grid.uuid)

    def _do_work(self):
        """Worker code to execute in a separate thread."""
        self.logger.info('Retrieving source grid data from SMS.')
        query_helper = SimQueries(self.sim_comp, self.query)
        # Request the required source data from XMS
        self.grid_data = query_helper.get_grid_mapping_data()
        if self.grid_data['error']:  # Had a problem, abort and try to unlink the source coverage
            if self.grid_data['sim_uuid'] and self.grid_data['grid_cov']:
                self.query.unlink_item(self.grid_data['sim_uuid'], self.grid_data['grid_cov'].uuid)
            return

        # Perform the mapping
        try:
            # Apply the 1D grid coverage to the simulation
            do_comp, py_comp = map_grid(self.query, self.grid_data)
            if do_comp and not LogEchoQSignalStream.logged_error:  # All done, no errors
                self._add_mapped_data(do_comp, py_comp)
                self.logger.info('1D GenCade grid coverage successfully applied to GenCade simulation.')
            else:
                raise RuntimeError()
        except Exception:  # Had a problem, abort and try to unlink the source coverage
            if self.grid_data['sim_uuid'] and self.grid_data['grid_cov']:
                self.query.unlink_item(self.grid_data['sim_uuid'], self.grid_data['grid_cov'].uuid)
            self.logger.exception('Unable to apply 1D grid to GenCade simulation.')
