"""A writer for GenCade simulation files."""

# 1. Standard Python modules
import bisect
from datetime import datetime
from io import StringIO
import logging
import os
import shutil

# 2. Third party modules
import numpy as np
from shapely.geometry import LineString, Point as shPoint
from shapely.ops import nearest_points

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.data_objects.parameters import FilterLocation
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.gencade.components.mapped_grid_component import MappedGridComponent
from xms.gencade.components.points_component import PointsComponent
from xms.gencade.components.struct_component import StructComponent
from xms.gencade.data.sim_data import SimData
from xms.gencade.data.struct_data import SHOAL_TYPES, TRANSMISSION_TYPES


class GenFileExporter:
    """A class for writing out a Gencade .gen file."""

    def __init__(self):
        """Constructor."""
        self._logger = logging.getLogger('xms.gencade')
        self.ss = StringIO()
        self.sim_name = ''
        self.proj_name = ''
        self.proj_dir = ''
        self.gen_file = ''
        self.inifile = ''
        self.dxfile = ''
        self.regfile = ''
        self.wlfile = ''
        self.secondary_file = ''
        self.location_file = ''

        # sim and coverage items
        self._sim_item = None
        self._sim_data = None
        self._sim_title = ''
        self._struct_data = None
        self._struct_comp = None
        self._struct_cov = None
        self._point_cov = None
        self._point_comp = None

        self._cov_mapper = None
        self._sms_version = ''
        self._query = None

        # grid
        self._grid_begin_pt = None
        self._mapped_grid = None
        self._grid_ls = None
        self._grid_dxs = []
        self._variable_dx = False
        self._dist_along_grid = []

        # structures
        self._init_shoreline = None
        self._reg_contour = None
        self._waves = dict()
        self._tides = dict()
        self._k1_mods = dict()
        self._k2_mods = dict()
        self._angle_amp_mods = dict()
        self._angle_inc_mods = dict()
        self._height_amp_mods = dict()
        self._berm_ht_mods = dict()
        self._depth_closure_mods = dict()
        self._diffracting_groins = []
        self._nondiffracting_groins = []
        self._breakwaters = []
        self._seawalls = []
        self._beachfill_events = []
        self._bypass_events = []
        self._inlets = []
        self._left_jetties = []
        self._right_jetties = []
        self._attachment_bars = []
        self._sbas_poly = []
        self._sbas_flux = []
        self._print_table = None
        self._arc_locs = []
        self._pt_locs = []

    def _get_offset_cell_id(self, starting_cell, distance, is_left):
        """Get the closest 1d grid cell to the given point.

        Args:
            starting_cell (int): The cell we are starting at.
            distance (float): Distance to travel from starting point.
            is_left (bool): True if moving to the left (smaller cell id's), False if to the right.

        Returns:
            cell_id (int): The corresponding cell id

        """
        if is_left and starting_cell == 0:
            return starting_cell
        elif not is_left and starting_cell == len(self._dist_along_grid) - 1:
            return starting_cell

        start_dist_along_grid = self._dist_along_grid(starting_cell)
        desired_dist = (start_dist_along_grid - distance) if is_left else (start_dist_along_grid + distance)
        index = bisect.bisect_left(self._dist_along_grid, desired_dist)
        return index

    def _get_cell_id_for_point(self, point):
        """Get the closest 1d grid cell to the given point.

        Args:
            point (Point): The point.

        Returns:
            cell_id (int): The corresponding cell id

        """
        nearest = nearest_points(self._grid_ls, shPoint(point.x, point.y, point.z))
        dist_along_line = self._grid_ls.project(nearest[0])
        index = bisect.bisect_left(self._dist_along_grid, dist_along_line)
        return index + 1

    def _get_dist_from_grid(self, pt):
        """Get the closest 1d grid cell to the given point.

        Args:
            pt (list): x,y,z location to determine the distance from the grid
        Returns
            distance (float): Distance from "pt" to the grid

        """
        return self._grid_ls.distance(shPoint(pt))

    def _get_simulation(self):
        """Get the simulation from SMS."""
        # Get the simulation tree item and get the hidden component data.
        sim_uuid = self._query.current_item_uuid()
        sim_comp = self._query.item_with_uuid(sim_uuid, model_name='GenCade', unique_name='SimComponent')

        sim_main_file = sim_comp.main_file
        self._sim_data = SimData(sim_main_file)
        self._sim_title = self._sim_data.model.title
        if self._sim_data.model.attrs['start_date'] == self._sim_data.model.attrs['end_date']:
            self._logger.error('Error: Start Date and End Date are the same in GenCade Model Control. Aborting.')
            return False
        self._sim_item = tree_util.find_tree_node_by_uuid(self._query.project_tree, sim_uuid)
        self.sim_name = self._sim_item.name
        return True

    def _get_sim_grid(self):
        """Get simulation grid from SMS."""
        # Get the grid info.
        mapped_grid_item = tree_util.descendants_of_type(self._sim_item, only_first=True,
                                                         unique_name='MappedGridComponent')
        if mapped_grid_item is None:
            self._logger.error('Error: No GenCade grid coverage has been linked to the simulation. Aborting.')
            return False

        self._mapped_grid = MappedGridComponent(mapped_grid_item.main_file)
        self._grid_dxs = self._mapped_grid.data.locations.dx.values
        self._variable_dx = len(np.unique(self._grid_dxs)) > 1
        self._dist_along_grid = [0]
        self._dist_along_grid_mid_pts = []
        for dx in self._grid_dxs:
            self._dist_along_grid.append(self._dist_along_grid[-1] + dx)
            self._dist_along_grid_mid_pts.append(self._dist_along_grid[-2] + (dx / 2.0))

        begin_pt = [self._mapped_grid.data.info.attrs['x0'], self._mapped_grid.data.info.attrs['y0']]
        end_pt = [self._mapped_grid.data.info.attrs['xend'], self._mapped_grid.data.info.attrs['yend']]
        self._grid_ls = LineString([begin_pt, end_pt])
        return True

    def _get_points_data(self):
        """Get points data from SMS."""
        # Get the linked points coverage and hidden component data.
        point_cov_item = tree_util.descendants_of_type(self._sim_item, xms_types=['TI_COVER_PTR'], allow_pointers=True,
                                                       coverage_type='GenCade Point Attributes', recurse=False,
                                                       only_first=True)

        if point_cov_item:
            self._point_cov = self._query.item_with_uuid(point_cov_item.uuid)
            point_do_comp = self._query.item_with_uuid(point_cov_item.uuid, model_name='GenCade',
                                                       unique_name='PointsComponent')
            self._point_comp = PointsComponent(point_do_comp.main_file)
            if self._point_comp.cov_uuid == '':
                self._point_comp.cov_uuid = self._point_cov.uuid
            self._point_comp.refresh_component_ids(self._query, points=True)

            self._points_data = self._point_comp.data
            cov_points = self._point_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
            for point in cov_points:
                comp_id = self._point_comp.get_comp_id(TargetType.point, point.id)
                if comp_id is None or comp_id < 0:
                    continue
                single_point = self._points_data.points.loc[dict(comp_id=[comp_id])]
                cell_id = self._get_cell_id_for_point(point)
                point_type = single_point['point_type'].item()

                # Try to keep track of the XY locations and types for all arcs and points for saving later to file.
                self._pt_locs.append({'location': point, 'type': point_type})

                if point_type == 'Wave Gage':
                    self._waves[cell_id] = single_point
                elif point_type == 'Attribute Modification':
                    if single_point['set_k1'].item() == 1:
                        self._k1_mods[cell_id] = single_point['k1_val'].item()
                    if single_point['set_k2'].item() == 1:
                        self._k2_mods[cell_id] = single_point['k2_val'].item()
                    if single_point['set_ang_amplify'].item() == 1:
                        self._angle_amp_mods[cell_id] = single_point['ang_amplify_val'].item()
                    if single_point['set_ang_adjust'].item() == 1:
                        self._angle_inc_mods[cell_id] = single_point['ang_adjust_val'].item()
                    if single_point['set_hgt_amplify'].item() == 1:
                        self._height_amp_mods[cell_id] = single_point['hgt_amplify_val'].item()
                    if single_point['set_berm_height'].item() == 1:
                        self._berm_ht_mods[cell_id] = single_point['berm_height_val'].item()
                    if single_point['set_depth_closure'].item() == 1:
                        self._depth_closure_mods[cell_id] = single_point['depth_closure_val'].item()
                elif point_type == 'Tidal Currents':
                    self._tides[cell_id] = single_point

    def _get_struct_data(self):
        """Get struct data from SMS."""
        # Get the linked points coverage and hidden component data.
        struct_cov_item = tree_util.descendants_of_type(self._sim_item, xms_types=['TI_COVER_PTR'], allow_pointers=True,
                                                        coverage_type='GenCade Structures-Events', recurse=False,
                                                        only_first=True)
        if struct_cov_item:
            self._struct_cov = self._query.item_with_uuid(struct_cov_item.uuid)
            struct_do_comp = self._query.item_with_uuid(struct_cov_item.uuid, model_name='GenCade',
                                                        unique_name='StructComponent')
            self._struct_comp = StructComponent(struct_do_comp.main_file)
            if self._struct_comp.cov_uuid == '':
                self._struct_comp.cov_uuid = self._struct_cov.uuid
            self._struct_comp.refresh_component_ids(self._query, arcs=True)

            self._struct_data = self._struct_comp.data
            cov_arcs = self._struct_cov.arcs
            for arc in cov_arcs:
                comp_id = self._struct_comp.get_comp_id(TargetType.arc, arc.id)
                if comp_id is None or comp_id < 0:
                    continue
                single_arc = self._struct_data.arcs.loc[dict(comp_id=[comp_id])]

                arc_type = single_arc['struct_type_cbx'].item()
                start_cell = self._get_cell_id_for_point(arc.start_node)
                end_cell = self._get_cell_id_for_point(arc.end_node)
                arc_points = arc.get_points(FilterLocation.PT_LOC_ALL)
                if start_cell < end_cell:
                    arc.start_node, arc.end_node = arc.end_node, arc.start_node
                    arc_points.reverse()

                pts = [[pt.x, pt.y, pt.z] for pt in arc_points]
                ls = LineString(pts)
                if start_cell < end_cell:
                    arc_item = {'start_cell': start_cell, 'end_cell': end_cell, 'start_loc': pts[0], 'end_loc': pts[-1],
                                'props': single_arc, 'linestring': ls}
                else:
                    arc_item = {'start_cell': end_cell, 'end_cell': start_cell, 'start_loc': pts[-1], 'end_loc': pts[0],
                                'props': single_arc, 'linestring': ls}
                # Try to keep track of the XY locations and types for all arcs and points for saving later to file.
                # No need to save initial shoreline or regional contours
                if arc_type not in ['Initial Shoreline', 'Regional Contour']:
                    # Seawalls have multiple vertices to keep track of, others are generally just straight lines.
                    if arc_type == 'Seawall' or arc_type == 'SBAS Polygon' or arc_type == 'SBAS Flux':
                        for i in range(len(pts) - 1):
                            self._arc_locs.append({'start_loc': pts[i], 'end_loc': pts[i + 1], 'type': arc_type,
                                                   'idx': i + 1})
                    else:
                        self._arc_locs.append({'start_loc': pts[0], 'end_loc': pts[-1], 'type': arc_type})

                if arc_type == 'Groin':
                    if single_arc['diffracting_chk'].item() == 1:
                        self._diffracting_groins.append(arc_item)
                    else:
                        self._nondiffracting_groins.append(arc_item)
                elif arc_type == 'Breakwater':
                    self._breakwaters.append(arc_item)
                elif arc_type == 'Seawall':
                    cells = [self._get_cell_id_for_point(pt) for pt in arc_points]
                    arc_item['cells'] = cells
                    arc_item['points'] = pts
                    self._seawalls.append(arc_item)
                elif arc_type == 'Beach Fill Event':
                    self._beachfill_events.append(arc_item)
                elif arc_type == 'Bypass Event':
                    self._bypass_events.append(arc_item)
                elif arc_type == 'Inlet':
                    self._inlets.append(arc_item)
                elif arc_type == 'Left Jetty on Inlet':
                    self._left_jetties.append(arc_item)
                elif arc_type == 'Right Jetty on Inlet':
                    self._right_jetties.append(arc_item)
                elif arc_type == 'Attachment Bar':
                    self._attachment_bars.append(arc_item)
                elif arc_type == 'Initial Shoreline':
                    self._init_shoreline = arc_item
                elif arc_type == 'Regional Contour':
                    self._reg_contour = arc_item
                elif arc_type == 'SBAS Polygon':
                    cells = [self._get_cell_id_for_point(pt) for pt in arc_points]
                    arc_item['cells'] = cells
                    self._sbas_poly.append(arc_item)
                elif arc_type == 'SBAS Flux':
                    cells = [self._get_cell_id_for_point(pt) for pt in arc_points]
                    min_dist = self._get_dist_from_grid(pts[0])
                    arc_item['cells'] = cells[0]
                    for i, pt in enumerate(pts[1:], 1):
                        dist = self._get_dist_from_grid(pt)
                        if dist < min_dist:
                            min_dist = dist
                            arc_item['cells'] = cells[i]
                    self._sbas_flux.append(arc_item)
        else:
            self._logger.error('Error: No GenCade structure-event coverage has been linked to the simulation. '
                               'Aborting.')
            return False
        return True

    def _map_struct_data(self):
        """Write struct data."""
        for lj in self._left_jetties:
            lj['min_cell'] = min(lj['start_cell'], lj['end_cell'])
            lj['max_cell'] = max(lj['start_cell'], lj['end_cell'])
            lj['max_dist'] = max(self._get_dist_from_grid(lj['start_loc']), self._get_dist_from_grid(lj['end_loc']))

        for rj in self._right_jetties:
            rj['min_cell'] = min(rj['start_cell'], rj['end_cell'])
            rj['max_cell'] = max(rj['start_cell'], rj['end_cell'])
            rj['max_dist'] = max(self._get_dist_from_grid(rj['start_loc']), self._get_dist_from_grid(rj['end_loc']))

        for inlet in self._inlets:
            if inlet['start_cell'] > inlet['end_cell']:
                # make sure that the start cell is the first one going along the grid
                inlet['start_cell'], inlet['end_cell'] = inlet['end_cell'], inlet['start_cell']

            inlet_start = inlet['start_cell']
            inlet_end = inlet['end_cell']

            # map left jetties
            inlet['left_jetty'] = None
            for l_jetty in self._left_jetties:
                if inlet_start + 2 >= l_jetty['min_cell'] and inlet_start - 1 <= l_jetty['max_cell']:
                    inlet['left_jetty'] = l_jetty
                    break

            # map right jetties
            inlet['right_jetty'] = None
            for r_jetty in self._right_jetties:
                if inlet_end + 2 >= r_jetty['min_cell'] and inlet_end - 1 <= r_jetty['max_cell']:
                    inlet['right_jetty'] = r_jetty
                    break

            # map attachment bars
            inlet['l_bar'] = None
            inlet['r_bar'] = None
            for bar in self._attachment_bars:
                if bar['start_cell'] <= inlet_start:
                    if inlet['l_bar'] is None:
                        inlet['l_bar'] = bar
                    else:
                        old_diff = inlet_start - inlet['l_bar']['start_cell']
                        new_diff = inlet_start - bar['start_cell']

                        # if this attachment bar is closer, use that
                        if new_diff < old_diff:
                            inlet['l_bar'] = bar
                elif bar['end_cell'] >= inlet_end:
                    if inlet['r_bar'] is None:
                        inlet['r_bar'] = bar
                    else:
                        old_diff = inlet['r_bar']['end_cell'] - inlet_end
                        new_diff = bar['end_cell'] - inlet_end

                        # if this attachment bar is closer, use that
                        if new_diff < old_diff:  # This line was missing, was it intended to be here?
                            inlet['r_bar'] = bar

    def _write_file_references(self):
        """Write file references."""
        # Build file names from project name for references in .gen file
        self.gen_file = f'{self.proj_name}.gen'
        self.inifile = f'{self.proj_name}.shi'
        self.dxfile = f'{self.proj_name}.shdx'
        self.regfile = f'{self.proj_name}.shr'
        self.wlfile = f'{self.proj_name}.wl'
        self.secondary_file = f'{self.proj_name}.cntmcxsh'
        self.location_file = f'{self.proj_name}.locs'

        self.ss.write('****** FILES ******\n')
        self.ss.write(f'PROJDIR:    "{self.proj_dir}"\n')
        self.ss.write(f'INIFILE:    "{self.inifile}"\n')
        if self._variable_dx:
            self.ss.write(f'DXFILE:     "{self.dxfile}"\n')
        self.ss.write(f'REGFILE:    "{self.regfile}"\n')
        self.ss.write(f'NUMWAVES:   {len(self._waves)}\n')
        wave_ids = sorted(self._waves.keys())
        for i, wave_id in enumerate(wave_ids):
            wave_depth = self._waves[wave_id]['wave_depth'].item()
            wave_file = f'{self.proj_name}_wave{i + 1}.wave'
            self._waves[wave_id]['wave_file'] = wave_file
            num_ts = len(self._point_comp.data.wave_table_from_id(self._waves[wave_id]['wave_table'])['wave_date'])
            self.ss.write(f'WAVEID:     {wave_id} {wave_depth:.2f} {num_ts} "{wave_file}"\n')

        self._write_gen_water_level()

        tide_ids = sorted(self._tides.keys())
        for i, tide_id in enumerate(tide_ids):
            tide_file = f'{self.proj_name}_tide{i + 1}.tide'
            self._tides[tide_id]['tide_file'] = tide_file
            num_ts = len(self._point_comp.data.tide_table_from_id(self._tides[tide_id]['tide_table'])['tide_date'])
            self.ss.write(f'TIDEFILE:   {tide_id} {num_ts} "{tide_file}"\n')
        self.ss.write(f'PRFILE:     "{self.proj_name}.prt"\n\n')

    def _write_with_mods(self, card, orig_val, mod_dict, end_card):
        """Write card with modifications.

        Args:
            card (str): the card to write
            orig_val (object): value for the card
            mod_dict (dict): dictionary of the modifications
            end_card (str): end card
        """
        self.ss.write(f'{card}{orig_val}')
        # write the mods if there are any
        mod_ids = sorted(mod_dict.keys())
        for mod_id in mod_ids:
            self.ss.write(f',{mod_dict[mod_id]}')
        if len(mod_ids) > 0:
            self.ss.write(f',{orig_val}')
        self.ss.write('\n')
        if len(mod_ids) > 0:
            self.ss.write(f'{end_card}1')
            for mod_id in mod_ids:
                self.ss.write(f',{mod_id}')
            self.ss.write(f',{len(self._grid_dxs)}\n')

    def _write_model_setup(self):
        """Write model setup section."""
        self.ss.write('***** MODEL SETUP *****\n')
        units = '(ft)' if 'feet' in self._query.display_projection.horizontal_units.lower() else '(m)'
        self.ss.write(f'GENUNITS:   {units}\n')  # Eventually need to get units figured out (either imperial or metric)
        self.ss.write(f'X0:         {self._mapped_grid.data.info.attrs["x0"]:.6f}\n')
        self.ss.write(f'Y0:         {self._mapped_grid.data.info.attrs["y0"]:.6f}\n')
        self.ss.write(f'AZIMUTH:    {self._mapped_grid.data.info.attrs["theta"].item():.2f}\n')
        self.ss.write(f'NX:         {len(self._grid_dxs)}\n')
        self.ss.write(f'DX:         {-1 if self._variable_dx else self._grid_dxs[0]}\n')
        date_time = datetime.fromisoformat(self._sim_data.model.start_date)
        date_str = datetime.strftime(date_time, '%Y%m%d')
        self.ss.write(f'SIMDATS:    {date_str}\n')
        date_time = datetime.fromisoformat(self._sim_data.model.end_date)
        date_str = datetime.strftime(date_time, '%Y%m%d')
        self.ss.write(f'SIMDATE:    {date_str}\n')
        self.ss.write(f'DT:         {self._sim_data.model.time_step}\n')
        self.ss.write(f'DTSAVE:     {self._sim_data.model.recording}\n')
        self._write_with_mods('K1:         ', self._sim_data.beach.K1, self._k1_mods, 'IK1END:     ')
        self._write_with_mods('K2:         ', self._sim_data.beach.K2, self._k2_mods, 'IK2END:     ')
        self.ss.write(f'KTIDE:      {self._sim_data.beach.KTIDE}\n')
        prtout = 't' if self._sim_data.model.full_print == 1 else 'f'
        self.ss.write(f'PRTOUT:     {prtout}\n')
        self.ss.write('PRWARN:     f\n')  # not supported?
        self._write_print_table_data()

        self.ss.write(f'ISMOOTH:    {self._sim_data.seaward.attrs["num_cells_smoothing"]}\n')
        self.ss.write('IREG:       1\n')  # always write out regional contour file

    def _write_adaptive_dt(self):
        """Write adaptive dt section."""
        if self._sim_data.adaptive.use_adaptive_ts == 0:
            return

        self.ss.write('***** Adaptive DT *****\n')
        self.ss.write(f'DTADAPT:    {self._sim_data.adaptive.use_adaptive_ts}\n')
        self.ss.write(f'DTMIN:      {self._sim_data.adaptive.threshold_min}\n')
        self.ss.write(f'DTMAX:      {self._sim_data.adaptive.threshold_max}\n')
        self.ss.write(f'NSTABDTMIN: {self._sim_data.adaptive.days_stable}\n')
        self.ss.write(f'STABMIN:    {self._sim_data.adaptive.stability_min}\n')
        self.ss.write(f'STABMAX:    {self._sim_data.adaptive.stability_max}\n')

    def _write_waves(self):
        """Write waves information."""
        self.ss.write('***** WAVES *****\n')
        self._write_with_mods('HAMP:       ', self._sim_data.seaward.ht_amp_factor,
                              self._height_amp_mods, 'IHAMPEND:   ')
        self._write_with_mods('THETAAMP:   ', self._sim_data.seaward.ang_amp_factor,
                              self._angle_amp_mods, 'ITHETAAMPEND:  ')
        self._write_with_mods('THETADEL:   ', self._sim_data.seaward.ang_offset,
                              self._angle_inc_mods, 'ITHETADELEND:  ')

        # These can all be removed. Not implemented.
        # missing GAMMA, WAVREGANG, IWAVREGSMOOTH, WAVUNITS

    def _write_beach(self):
        """Write beach information."""
        # Note: I just looked in the code and Gated can be either 1 or 2 depending on the closest groin or jetty
        # Gated = 1 when a groin to the left or right is within the threshold
        # Gated = 2 when a jetty to the left or right is within the threshold
        # Otherwise Pinned is always 0 and Moving is always 3
        lateral_type = {'Pinned': 0, 'Gated': 2, 'Moving': 3}
        lateral_movper = {'Simulation Period': 1, 'Day': 2, 'Timestep': 3}
        self.ss.write('***** BEACH *****\n')
        self.ss.write(f'D50:        {self._sim_data.beach.eff_grain_size}\n')
        self._write_with_mods('BERMHT:     ', self._sim_data.beach.avg_berm_ht,
                              self._berm_ht_mods, 'IBERMHTEND: ')
        self._write_with_mods('DCLOS:      ', self._sim_data.beach.closure_depth,
                              self._depth_closure_mods, 'IDCLOSEND:  ')
        self.ss.write(f'LBCTYPE:    {lateral_type[self._sim_data.lateral.left_bc_type]}\n')
        self.ss.write(f'LMOVY:      {self._sim_data.lateral.left_shoreline_disp_distance}\n')
        self.ss.write(f'LMOVPER:    {lateral_movper[self._sim_data.lateral.left_disp_type]}\n')
        self.ss.write(f'LGROINY:    {self._sim_data.lateral.left_length_to_tip}\n')
        self.ss.write(f'RBCTYPE:    {lateral_type[self._sim_data.lateral.rt_bc_type]}\n')
        self.ss.write(f'RMOVY:      {self._sim_data.lateral.rt_shoreline_disp_distance}\n')
        self.ss.write(f'RMOVPER:    {lateral_movper[self._sim_data.lateral.rt_disp_type]}\n')
        self.ss.write(f'RGROINY:    {self._sim_data.lateral.rt_length_to_tip}\n')

    def _write_groins(self):
        """Write groins information."""
        if len(self._nondiffracting_groins) == 0 and len(self._diffracting_groins) == 0:
            return

        self.ss.write('***** GROINS *****\n')
        for groin in self._nondiffracting_groins:
            self.ss.write(f'IXNDG:      {groin["start_cell"]}\n')
            # self.ss.write(f'YNDG:       {groin["linestring"].length}\n')
            self.ss.write(f'YNDG:       {self._get_dist_from_grid(groin["end_loc"]):.6f}\n')
            self.ss.write(f'PNDG:       {groin["props"]["groin_permeability"].item()}\n')

        # Diffracting groins
        for groin in self._diffracting_groins:
            self.ss.write(f'IXDG:       {groin["start_cell"]}\n')
            # self.ss.write(f'YDG:        {groin["linestring"].length}\n')
            self.ss.write(f'YDG:        {self._get_dist_from_grid(groin["end_loc"]):.6f}\n')
            self.ss.write(f'DDG:        {groin["props"]["seaward_depth"].item()}\n')
            self.ss.write(f'PDG:        {groin["props"]["groin_permeability"].item()}\n')

    def _write_detached_breakwaters(self):
        """Write detached breakwater information."""
        if len(self._breakwaters) == 0:
            return

        self.ss.write('***** DETACHED BREAKWATERS *****\n')
        for bkw in self._breakwaters:
            self.ss.write(f'DBI1:       {bkw["start_cell"]}\n')
            self.ss.write(f'DBY1:       {self._get_dist_from_grid(bkw["start_loc"]):.6f}\n')
            self.ss.write(f'DBI2:       {bkw["end_cell"]}\n')
            self.ss.write(f'DBY2:       {self._get_dist_from_grid(bkw["end_loc"]):.6f}\n')
            self.ss.write(f'DBDEP1:     {bkw["props"]["depth1"].item()}\n')
            self.ss.write(f'DBDEP2:     {bkw["props"]["depth2"].item()}\n')
            if bkw["props"]["transmission_type_cbx"].item() == 'Constant':
                self.ss.write(f'TRANDB:     {bkw["props"]["transmission_const"].item()}\n')
            else:
                method = TRANSMISSION_TYPES.index(bkw["props"]["transmission_type_cbx"].item()) - 1
                self.ss.write(f'KTMETDB:     {method}\n')
                self.ss.write(f'HDB:         {bkw["props"]["height"].item()}\n')
                self.ss.write(f'BDB:         {bkw["props"]["width"].item()}\n')
                self.ss.write(f'SSDB:        {bkw["props"]["seaward_slope"].item()}\n')
                self.ss.write(f'SHSDB:       {bkw["props"]["shoreward_slope"].item()}\n')
                if method < 2:
                    self.ss.write(f'D50DB:       {bkw["props"]["armor_d50"].item()}\n')
                if method == 2:
                    self.ss.write(f'PDB:         {bkw["props"]["bw_permeability"].item()}\n')
        # missing IDB1          can be removed, not presently used
        # missing IDBN          can be removed, not presently used
        # missing DWTL          will try to figure this one out

    def _write_seawalls(self):
        """Write seawall information."""
        if len(self._seawalls) == 0:
            return

        self.ss.write('***** SEAWALL *****\n')
        for sw in self._seawalls:
            for i in range(len(sw['cells']) - 1):  # Write Cells and offsets for all segments of arc.
                self.ss.write(f'ISWBEG:     {sw["cells"][i]}\n')
                self.ss.write(f'ISWEND:     {sw["cells"][i+1]}\n')
                self.ss.write(f'SWY1:       {self._get_dist_from_grid(sw["points"][i]):.6f}\n')
                self.ss.write(f'SWY2:       {self._get_dist_from_grid(sw["points"][i+1]):.6f}\n')

    def _write_sbas(self):
        """Write sbas polygon and flux information."""
        if len(self._sbas_poly) == 0 and len(self._sbas_flux) == 0:
            return

        self.ss.write('***** SBAS *****\n')
        poly_list = []
        flux_cells = set()
        # Gather first
        for sbas in self._sbas_poly:
            cells = sorted(sbas['cells'])  # Sort numbers so they start at grid origin and work down the line.
            poly_list.extend((cells[i], cells[i + 1] - 1) for i in range(len(cells) - 1))
            flux_cells.update(cells)
        for sbas in self._sbas_flux:
            # Gather standalone flux cells closest to grid
            flux_cells.add(sbas['cells'])  # Append Cell for all segments of flux arc.

        # Then write them out
        for start, end in poly_list:
            self.ss.write(f'ISBASV1:    {start}\n'
                          f'ISBASV2:    {end}\n')
        for cell in sorted(flux_cells):
            self.ss.write(f'ISBASQ1:    {cell}\n'
                          f'ISBASQ2:    {cell}\n')

    def _write_beachfill(self):
        """Write beachfill information."""
        if len(self._beachfill_events) == 0:
            return

        self.ss.write('***** BEACHFILL *****\n')
        for bf in self._beachfill_events:
            table_id = bf['props']['beach_fill_table'].item()
            grp = self._struct_data.beach_fill_table_group_name(table_id)
            table = self._struct_data.get_dataset(grp, False)

            # I can't get the datetime correctly from the dataset, but it works from a Pandas DataFrame
            pandas = table.to_dataframe()
            date_time = datetime.fromisoformat(str(pandas['start_date'].item()))
            date_str = datetime.strftime(date_time, '%Y%m%d')
            self.ss.write(f'BFDATS:     {date_str}\n')
            date_time = datetime.fromisoformat(str(pandas['end_date'].item()))
            date_str = datetime.strftime(date_time, '%Y%m%d')
            self.ss.write(f'BFDATE:     {date_str}\n')
            self.ss.write(f'IBFS:       {bf["start_cell"]}\n')
            self.ss.write(f'IBFE:       {bf["end_cell"]}\n')
            self.ss.write(f'YADD:       {pandas["added_berm_width"].item()}\n')

    def _write_bypass(self):
        """Write bypass information."""
        if len(self._bypass_events) == 0:
            return

        self.ss.write('***** BYPASS *****\n')
        for bp in self._bypass_events:
            table_id = bp['props']['bypass_table'].item()
            grp = self._struct_data.bypass_table_group_name(table_id)
            table = self._struct_data.get_dataset(grp, False)

            # I can't get the datetime correctly from the dataset, but it works from a Pandas DataFrame
            pandas = table.to_dataframe()
            for idx in range(pandas.shape[0]):
                date_time = datetime.fromisoformat(str(pandas['start_date'][idx + 1]))
                date_str = datetime.strftime(date_time, '%Y%m%d')
                self.ss.write(f'BPDATS:     {date_str}\n')
                date_time = datetime.fromisoformat(str(pandas['end_date'][idx + 1]))
                date_str = datetime.strftime(date_time, '%Y%m%d')
                self.ss.write(f'BPDATE:     {date_str}\n')
                self.ss.write(f'IBPS:       {bp["start_cell"]}\n')
                self.ss.write(f'IBPE:       {bp["end_cell"]}\n')
                self.ss.write(f'QBP:        {pandas["bypass_rate"][idx + 1]}\n')

    def _write_inlets(self):
        """Write inlet information."""
        if len(self._inlets) == 0:
            return

        self.ss.write('***** INLETS *****\n')
        for inlet in self._inlets:
            inlet_start = inlet['start_cell']
            inlet_end = inlet['end_cell']
            self.ss.write(f'IINLS:      {inlet_start}\n')
            self.ss.write(f'IINLE:      {inlet_end}\n')
            self.ss.write(f'INLNAME:    {inlet["props"]["name"].item()}\n')

            # write the left bypass location info
            if inlet['l_bar']:
                # don't allow the cells to overlap the inlet
                start = min(inlet_start - 1, inlet['l_bar']['start_cell'])
                end = min(inlet_start - 1, inlet['l_bar']['end_cell'])
                self.ss.write(f'IBYPL1:     {start}\n')
                self.ss.write(f'IBYPL2:     {end}\n')
            else:
                # if there is no attachment bar, just make the bypass one cell wide at the end
                self.ss.write(f'IBYPL1:     {inlet_start - 1}\n')
                self.ss.write(f'IBYPL2:     {inlet_start - 1}\n')

            # write the right bypass location info
            if inlet['r_bar']:
                # don't allow the cells to overlap the inlet
                start = max(inlet_end + 1, inlet['r_bar']['start_cell'])
                end = max(inlet_end + 1, inlet['r_bar']['end_cell'])
                self.ss.write(f'IBYPR1:     {start}\n')
                self.ss.write(f'IBYPR2:     {end}\n')
            else:
                # if there is no attachment bar, just make the bypass one cell wide at the end
                self.ss.write(f'IBYPR1:     {inlet_end + 1}\n')
                self.ss.write(f'IBYPR2:     {inlet_end + 1}\n')

            # volume info
            self.ss.write(f'VVSE:       {inlet["props"]["ebb_shoal_init"].item()}\n')
            self.ss.write(f'VVSF:       {inlet["props"]["flood_shoal_init"].item()}\n')
            self.ss.write(f'VVSBL:      {inlet["props"]["left_bypass_init"].item()}\n')
            self.ss.write(f'VVSBR:      {inlet["props"]["right_bypass_init"].item()}\n')
            self.ss.write(f'VVSAL:      {inlet["props"]["left_attach_init"].item()}\n')
            self.ss.write(f'VVSAR:      {inlet["props"]["right_attach_init"].item()}\n')
            self.ss.write(f'VVSEQ:      {inlet["props"]["ebb_shoal_equil"].item()}\n')
            self.ss.write(f'VVSFQ:      {inlet["props"]["flood_shoal_equil"].item()}\n')
            self.ss.write(f'VVSBQL:     {inlet["props"]["left_bypass_equil"].item()}\n')
            self.ss.write(f'VVSBQR:     {inlet["props"]["right_bypass_equil"].item()}\n')
            self.ss.write(f'VVSAQL:     {inlet["props"]["left_attach_equil"].item()}\n')
            self.ss.write(f'VVSAQR:     {inlet["props"]["right_attach_equil"].item()}\n')

            # jetties
            if inlet['left_jetty'] is not None:
                self.ss.write(f'JBCL:       {inlet["props"]["left_bypass_coeff"].item()}\n')
            if inlet['right_jetty'] is not None:
                self.ss.write(f'JBCR:       {inlet["props"]["right_bypass_coeff"].item()}\n')

            if inlet['left_jetty'] is not None:
                jetty = inlet['left_jetty']
                if jetty['props']['diffracting_chk'].item() == 1:
                    self.ss.write(f'IXDG:       {inlet["start_cell"]}\n')
                    self.ss.write(f'YDG:        {jetty["max_dist"]:.6f}\n')
                    self.ss.write(f'DDG:        {jetty["props"]["seaward_depth"].item()}\n')
                    self.ss.write(f'PDG:        {jetty["props"]["groin_permeability"].item()}\n')
                else:
                    self.ss.write(f'IXNDG:      {inlet["start_cell"]}')
                    self.ss.write(f'YNDG:       {jetty["max_dist"]:.6f}\n')
                    self.ss.write(f'PNDG:       {jetty["props"]["groin_permeability"].item()}\n')

            if inlet['right_jetty'] is not None:
                jetty = inlet['right_jetty']
                if jetty['props']['diffracting_chk'].item() == 1:
                    self.ss.write(f'IXDG:       {inlet["end_cell"]}\n')
                    self.ss.write(f'YDG:        {jetty["max_dist"]:.6f}\n')
                    self.ss.write(f'DDG:        {jetty["props"]["seaward_depth"].item()}\n')
                    self.ss.write(f'PDG:        {jetty["props"]["groin_permeability"].item()}\n')
                else:
                    self.ss.write(f'IXNDG:      {inlet["end_cell"]}')
                    self.ss.write(f'YNDG:       {jetty["max_dist"]:.6f}\n')
                    self.ss.write(f'PNDG:       {jetty["props"]["groin_permeability"].item()}\n')

            # write out the dredging info
            table_id = inlet['props']['dredging_table'].item()
            grp = self._struct_data.dredging_table_group_name(table_id)
            table = self._struct_data.get_dataset(grp, False)

            # I can't get the datetime correctly from the dataset, but it works from a Pandas DataFrame
            pandas = table.to_dataframe()
            for idx in range(pandas.shape[0]):
                # shoal method
                shoal_type = SHOAL_TYPES.index(pandas['shoal_to_be_mined'][idx + 1])  # + 1
                self.ss.write(f'IMOR:       {shoal_type}\n')
                # I don't think the + 1 is needed, because there is a '(none selected)' option in the combo box

                # start date
                begin_date = datetime.fromisoformat(str(pandas['begin_date'][idx + 1]))
                date_str = datetime.strftime(begin_date, '%Y%m%d')
                self.ss.write(f'IDREDS:     {date_str}\n')

                # end date
                end_date = datetime.fromisoformat(str(pandas['end_date'][idx + 1]))  # there was an error here, fixed
                date_str = datetime.strftime(end_date, '%Y%m%d')
                self.ss.write(f'IDREDE:     {date_str}\n')

                # number of days in event
                delta = end_date - begin_date
                self.ss.write(f'DRDAY:      {delta.days + 1}\n')

                # volume
                self.ss.write(f'DREDV:      {pandas["volume"][idx + 1]}\n')

    def _write_gen_water_level(self):
        """Write the water level data."""
        if self._sim_data.model.attrs['enable_water_level'] == 0:
            return

        wl_table = self._sim_data.wl_table.to_dataframe()
        num_ts = len(wl_table)
        if num_ts == 0:
            return

        self.ss.write(f'WLFILE:     {num_ts} "{self.wlfile}"\n')

    def _write_print_table_data(self):
        """Write the print table data."""
        pr_table = self._sim_data.print_table.to_dataframe()
        if len(pr_table) == 0:
            return

        # Special case if there is only one time and it is exactly 2000-01-01
        date_str = datetime.strftime(datetime.fromisoformat(str(pr_table['print_date'][1])), '%Y%m%d')
        if len(pr_table) == 1 and date_str == '20000101':
            return

        for idx in range(pr_table.shape[0]):
            date_str = datetime.strftime(datetime.fromisoformat(str(pr_table['print_date'][idx + 1])), '%Y%m%d')
            self.ss.write(f'PRDATE:     {date_str}\n')

    def _write_gen_file(self):
        """Write the general file."""
        self._logger.info('Writing general file...')
        self.ss.write('GENCADE:\n\n')
        self.ss.write(f'TITLE:    {self._sim_title}\n')

        self._write_file_references()
        self._write_model_setup()
        self._write_adaptive_dt()
        self._write_waves()
        self._write_beach()
        self._write_groins()
        self._write_detached_breakwaters()
        self._write_seawalls()
        self._write_beachfill()
        self._write_bypass()
        self._write_inlets()
        self._write_sbas()

        # flush to file
        out = open(self.gen_file, 'w')
        self.ss.seek(0)
        shutil.copyfileobj(self.ss, out)
        out.close()

    def _write_variable_grid_file(self):
        """Write the variable grid file."""
        if self._variable_dx is False:
            return

        self._logger.info('Writing variable grid file...')
        ss = StringIO()
        ss.write('************************************************\n')
        ss.write(' DX SHORELINE DATA FOR VARIABLE GRID FOR GENCADE:\n')
        ss.write(f' {len(self._grid_dxs)} Sizes - TITLE:{self._sim_title}\n')
        ss.write('************************************************\n')

        i_count = 0
        for dx in self._grid_dxs:
            i_count += 1
            ss.write(f'{dx:.3f} ')
            if i_count == 10:
                ss.write('\n')
                i_count = 0

        # flush to file
        out = open(self.dxfile, 'w')
        ss.seek(0)
        shutil.copyfileobj(ss, out)
        out.close()

    def _write_offsets_from_grid(self, ls, ss):
        """Write the offsets from the grid."""
        new_line_count = 0
        for mid_dist in self._dist_along_grid_mid_pts:
            new_line_count += 1
            ss.write(f'{self._grid_ls.interpolate(mid_dist).distance(ls):.3f} ')
            if new_line_count == 10:
                ss.write('\n')
                new_line_count = 0

    def _write_init_shoreline_file(self):
        """Write the initial shoreline file."""
        self._logger.info('Writing initial shoreline file...')
        ss = StringIO()
        ss.write('************************************************\n')
        ss.write(' INITIAL SHORELINE DATA FOR GENCADE:\n')
        ss.write(f' {len(self._grid_dxs)} Offsets - TITLE:{self._sim_title}\n')
        ss.write('************************************************\n')

        self._write_offsets_from_grid(self._init_shoreline['linestring'], ss)

        # flush to file
        out = open(self.inifile, 'w')
        ss.seek(0)
        shutil.copyfileobj(ss, out)
        out.close()

    def _write_reg_contour_file(self):
        """Write the regional contour file."""
        if self._reg_contour is None:
            return

        self._logger.info('Writing regional contour file...')
        ss = StringIO()
        ss.write('************************************************\n')
        ss.write(' REGIONAL SHORELINE DATA FOR GENCADE:\n')
        ss.write(f' {len(self._grid_dxs)} Offsets - TITLE:{self._sim_title}\n')
        ss.write('************************************************\n')

        self._write_offsets_from_grid(self._reg_contour['linestring'], ss)

        # flush to file
        out = open(self.regfile, 'w')
        ss.seek(0)
        shutil.copyfileobj(ss, out)
        out.close()

    def _write_wave_files(self):
        """Write the wave files."""
        if len(self._waves) == 0:
            return

        self._logger.info('Writing wave files...')
        for wave_id in self._waves.keys():
            ss = StringIO()
            wave_table = self._point_comp.data.wave_table_from_id(self._waves[wave_id]['wave_table'])
            wave_df = wave_table.to_dataframe()

            for idx in range(wave_df.shape[0]):
                date_time = datetime.fromisoformat(str(wave_df['wave_date'][idx + 1]))
                d_str = datetime.strftime(date_time, '%Y%m%d')
                t_str = datetime.strftime(date_time, '%H%M')
                height = wave_df['wave_height'][idx + 1]
                period = wave_df['wave_period'][idx + 1]
                direction = wave_df['wave_direction'][idx + 1]
                ss.write(f'{d_str} {t_str} {height:2f} {period:2f} {direction:2f}\n')

            wave_file_name = self._waves[wave_id]['wave_file'].item()
            out = open(wave_file_name, 'w')
            ss.seek(0)
            shutil.copyfileobj(ss, out)
            out.close()

    def _write_tide_files(self):
        """Write the tide files."""
        if len(self._tides) == 0:
            return

        self._logger.info('Writing tide files...')
        tide_ids = sorted(self._tides.keys())
        for tide_id in tide_ids:
            ss = StringIO()
            tide_table = self._point_comp.data.tide_table_from_id(self._tides[tide_id]['tide_table'])
            tide = tide_table.to_dataframe()

            for idx in range(tide.shape[0]):
                date_time = datetime.fromisoformat(str(tide['tide_date'][idx + 1]))
                d_str = datetime.strftime(date_time, '%Y%m%d')
                t_str = datetime.strftime(date_time, '%H%M')
                speed = tide['current_speed'][idx + 1]
                ss.write(f'{d_str} {t_str} {speed:2f}\n')

            tide_file_name = self._tides[tide_id]['tide_file'].item()
            out = open(tide_file_name, 'w')
            ss.seek(0)
            shutil.copyfileobj(ss, out)
            out.close()

    def _write_water_level_file(self):
        """Write the water level file."""
        if self._sim_data.model.attrs['enable_water_level'] == 0:
            return

        self._logger.info('Writing water level file...')
        ss = StringIO()

        wl_table = self._sim_data.wl_table.to_dataframe()
        for idx in range(wl_table.shape[0]):
            date_time = datetime.fromisoformat(str(wl_table['wl_date'][idx + 1]))
            d_str = datetime.strftime(date_time, '%Y%m%d')
            t_str = datetime.strftime(date_time, '%H%M')
            wl = wl_table['water_level'][idx + 1]
            ss.write(f'{d_str} {t_str} {wl:2f}\n')

        # flush to file
        out = open(self.wlfile, 'w')
        ss.seek(0)
        shutil.copyfileobj(ss, out)
        out.close()

    def _write_secondary_file(self):
        """Write the secondary file."""
        self._logger.info('Writing secondary file...')
        ss = StringIO()

        off_on = ['OFF', 'ON']
        if self._sim_data.cross_shore.attrs['use_cross_shore'] == 1:
            ss.write('!Cross Shore\n')
            ss.write('ENABLE_XSHORE_SED            ON\n')
            ss.write(f'ALPHA_XSHORE                 {self._sim_data.cross_shore.attrs["cross_shore_scale"]}\n')
            ss.write(f'CHANGE_BERMHEIGHT_BY_SLR     {off_on[self._sim_data.cross_shore.attrs["use_var_berm_ht"]]}\n')
            ss.write(f'CHANGE_BEACHFACE_SLOPE       {off_on[self._sim_data.cross_shore.attrs["use_var_slope"]]}\n')
            ss.write(f'DEFAULT_ONSHORE_RATE         {self._sim_data.cross_shore.attrs["onshore_trans_rate"]}\n\n')

        if self._sim_data.monte_carlo.attrs['use_monte_carlo'] == 1:
            wave_pdf = ['Rayleigh Distribution', 'Rayleigh+Weibull', 'User Specified']
            prob_index = self._sim_data.monte_carlo.attrs["prob_function"]
            use_beach_fill = self._sim_data.monte_carlo.attrs['use_beach_fill']

            ss.write('!Monte Carlo\n')
            ss.write('ENABLE_MONTECARLO            ON\n')
            # check on this next one
            ss.write(f'NRUNS_MONTECARLO             {int(self._sim_data.monte_carlo.attrs["num_mc_sims"])}\n')
            ss.write(f'WAVE_PDF                     {wave_pdf.index(prob_index) + 1}\n')

            ss.write(f'MEAN_ANGLE                   {self._sim_data.monte_carlo.attrs["AL_mean"]}\n')
            ss.write(f'STD_ANGLE                    {self._sim_data.monte_carlo.attrs["AL_sigma"]}\n')
            ss.write(f'MAX_WAVEANGLE                {self._sim_data.monte_carlo.attrs["AL_max"]}\n')
            if self._sim_data.monte_carlo.attrs['prob_function'] == 'Rayleigh+Weibull':
                ss.write(f'A0_WEIBULL                   {self._sim_data.monte_carlo.attrs["AAO_val"]}\n')
                ss.write(f'B0_WEIBULL                   {self._sim_data.monte_carlo.attrs["BBO_val"]}\n')
                ss.write(f'K0_WEIBULL                   {self._sim_data.monte_carlo.attrs["FKO_val"]}\n')
            if self._sim_data.monte_carlo.attrs['prob_function'] != 'User Specified':
                ss.write(f'HX_THRESHOLD                 {self._sim_data.monte_carlo.attrs["HCUT_val"]}\n')
                ss.write(f'MIN_WAVEHEIGHT               {self._sim_data.monte_carlo.attrs["H_min"]}\n')
                ss.write(f'MAX_WAVEHEIGHT               {self._sim_data.monte_carlo.attrs["H_max"]}\n')
                ss.write(f'MEAN_WAVEHEIGHT              {self._sim_data.monte_carlo.attrs["H_mean"]}\n')
                ss.write(f'RUN_MONTECARLO_BEACHFILL     {off_on[use_beach_fill]}\n')
                if use_beach_fill == 1:
                    ss.write(f'STDRATE_BEACHFILL_WIDTH      {self._sim_data.monte_carlo.attrs["std_percent"]}\n')
                ss.write('\n')

        ss.write('!Beach Properties\n')
        ss.write(f'SLR_RATE                     {self._sim_data.beach.regional_slc}\n')
        ss.write(f'SUBSIDENCE_RATE              {self._sim_data.beach.regional_subsidence}\n')
        ss.write(f'BEACHFACE_SLOPE              {self._sim_data.beach.beach_slope}\n')

        # flush to file
        out = open(self.secondary_file, 'w')
        ss.seek(0)
        shutil.copyfileobj(ss, out)
        out.close()

    def _write_locations_file(self):
        """Write the locations file."""
        if len(self._pt_locs + self._arc_locs) == 0:
            return

        self._logger.info('Writing locations file...')
        ss = StringIO()
        for pt in self._pt_locs:
            ss.write(f'POINT       {pt["location"].x:.3f} {pt["location"].y:.3f}  {pt["type"]}\n')

        for arc in self._arc_locs:
            if arc['type'] != 'Seawall':
                ss.write(f'ARC         {arc["start_loc"][0]:.3f} {arc["start_loc"][1]:.3f}  {arc["end_loc"][0]:.3f} '
                         f'{arc["end_loc"][1]:.3f}  {arc["type"]}\n')
            else:
                ss.write(f'MULT_ARC_{arc["idx"]}  {arc["start_loc"][0]:.3f} {arc["start_loc"][1]:.3f}  '
                         f'{arc["end_loc"][0]:.3f} {arc["end_loc"][1]:.3f}  {arc["type"]}\n')

        # flush to file
        out = open(self.location_file, 'w')
        ss.seek(0)
        shutil.copyfileobj(ss, out)
        out.close()

    def write(self):
        """Write the main GenCade file."""
        self._query = Query()
        self.proj_name = os.path.splitext(os.path.basename(self._query.xms_project_path))[0]
        self.proj_dir = os.getcwd()

        # Build file names from project name for references in .gen file
        self.gen_file = f'{self.proj_name}.gen'
        self.inifile = f'{self.proj_name}.shi'
        self.dxfile = f'{self.proj_name}.shdx'
        self.regfile = f'{self.proj_name}.shr'

        if not self._get_simulation() or not self._get_sim_grid() or not self._get_struct_data():
            return
        self._get_points_data()
        self._map_struct_data()

        self._write_gen_file()
        self._write_variable_grid_file()
        self._write_init_shoreline_file()
        self._write_reg_contour_file()
        self._write_wave_files()
        self._write_tide_files()
        self._write_water_level_file()
        self._write_secondary_file()
        self._write_locations_file()
