"""This is the base class table widget for other table widgets in GenCade."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QItemSelectionModel

# 3. Aquaveo modules
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.widgets.basic_table_widget import BasicTableWidget

# 4. Local modules


class GenCadeTableWidget(BasicTableWidget):
    """Table widget base class."""
    def __init__(self, parent, data_frame, select_col, default_values, time_format='%Y-%m-%d'):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (:obj:`pandas.DataFrame`): The model data.
            select_col (:obj:`int`): Column to select when adding/removing rows
            default_values (:obj:`dict`): Default DataFrame values for added rows
            time_format (str): Format to display time

        """
        super().__init__(parent)
        self.select_col = select_col
        self.default_values = default_values
        if data_frame is not None:
            if len(data_frame.index) > 0 and data_frame.index[0] == 0:
                data_frame.index += 1  # Make vertical header 1-based
            self.model = QxPandasTableModel(data_frame, self)
            self.model.time_format = time_format
            self.model.set_default_values(self.default_values)
        self.setup_ui()

    def on_btn_add_row(self):
        """Called when a new row is added to the table."""
        row_idx = self.model.rowCount()
        self.model.insertRows(row_idx, 1)
        new_index = self.model.index(row_idx, self.select_col)
        self.table_view.selectionModel().setCurrentIndex(
            new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
        )

    def on_btn_delete_row(self):
        """Called when a row is removed from the table."""
        indices = self.table_view.selectionModel().selectedIndexes()
        next_select_row = -1
        delete_rows = {index.row() if index.isValid() else -1 for index in indices}
        delete_count = 0
        for index in delete_rows:
            if index < 0:
                continue
            delete_row = index - delete_count
            next_select_row = delete_row - 1
            self.model.removeRows(delete_row, 1)
            delete_count += 1
        if next_select_row >= 0:
            select_index = self.table_view.model().index(next_select_row, self.select_col)
            self.table_view.selectionModel().setCurrentIndex(
                select_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
            )

    def setup_ui(self):
        """Sets up the widgets, delegates, etc..."""
        super()._setup_ui({}, False, False)
