"""This is a dialog for specifying structure arc attributes."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
import uuid
import webbrowser

# 2. Third party modules
from PySide2.QtWidgets import QLabel

# 3. Aquaveo modules
from xms.api.tree.tree_util import build_tree_path, ProjectExplorerTreeCreator
from xms.components.bases.xarray_base import try_to_compute_relative_path
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.validators.qx_int_validator import QxIntValidator

# 4. Local modules
from xms.gencade.data.struct_data import STRUCT_TYPES, TRANSMISSION_TYPES
from xms.gencade.gui.beach_fill_table_widget import BeachFillTableWidget
from xms.gencade.gui.bypass_table_widget import BypassTableWidget
from xms.gencade.gui.dredging_table_widget import DredgingTableWidget
from xms.gencade.gui.file_selector import FileSelector
from xms.gencade.gui.struct_event_dlg_ui import Ui_StructuresEventsDlg
from xms.gencade.gui.text_converter import TextConverter


def find_matching_index(str_list: list[str], substring: str) -> int | None:
    """Finds and returns the index of the first string that includes the substring.

    Args:
        str_list (:obj: `list`): the list of strings to compare with the substring.
        substring (:obj:`str`): the substring to compare.
    """
    for i, string in enumerate(str_list):
        if substring in string:
            return i
    return None


class StructureDlg(XmsDlg):
    """A dialog for viewing structure data for an arc."""

    def __init__(self, struct_data, all_data, comp_id, pe_tree, multi_msg, units, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            struct_data (:obj:`xarray.Dataset`): The structure data for the arc to view
            all_data (:obj:`StructData`): The structure data for all arcs.
            comp_id (:obj:`int`): The component id for this arc.
            pe_tree (:obj:`TreeNode`): The project explorer for selecting datasets
            multi_msg (:obj:`str`): Warning/multi-select message, if any
            units (:obj:`str`): Units as determined by a query in the component.
            parent (Something derived from :obj:`QWidget`): The parent window
        """
        super().__init__(parent, 'xms.gencade.gui.struct_event_dlg')
        self._help_url = 'https://cirpwiki.info/wiki/GenCade_2.0_Structures_Dialog'
        self._units = 'Imperial' if 'feet' in units else 'Metric'
        self.ui = Ui_StructuresEventsDlg()
        self.ui.setupUi(self)
        self.ui.units_label.setText(f'Units for this projection are {self._units}.')
        self.ui.units_label.setStyleSheet('color: red;')
        self.struct_data = struct_data
        self.all_data = all_data
        self.pe_tree = pe_tree
        self.petc = ProjectExplorerTreeCreator()
        self.multi_msg = multi_msg

        self.comp_id = comp_id

        self.dbl_validator = QxDoubleValidator(parent=self, decimals=10)
        self.nonzero_dbl_validator = QxDoubleValidator(parent=self, bottom=0.01)
        self.int_validator = QxIntValidator(parent=self)
        self.nonzero_dbl_validator = QxDoubleValidator(parent=self, bottom=0.01)
        self.int_validator = QxIntValidator(parent=self)

        # Set the project directory (if there is one) for resolving relative paths
        FileSelector.project_folder = self.all_data.info.attrs['proj_dir']

        self._add_multi_select_warning()
        self._add_tables()
        self._setup_combo_boxes()
        self._setup_edit_fields()

        self._connect_widgets()
        self._load_data()
        self.adjustSize()

    def _add_multi_select_warning(self):
        """Add a label for the multi-select warning, if applicable."""
        if self.multi_msg:
            label = QLabel(self.multi_msg)
            label.setStyleSheet('QLabel{color: rgb(255, 0, 0);}')
            self.layout().insertWidget(0, label)

    def _add_tables(self):
        """Adds tables to dialog for dredging, bypass, and beach fill events."""
        if not self.struct_data.name.size == 0:
            table_id = int(self.struct_data['dredging_table'].item())
            if table_id <= 0:
                table_id = self.comp_id
                self.struct_data['dredging_table'] = table_id
            dredging_table = self.all_data.dredging_table_from_id(table_id)
            self.dredging_table = DredgingTableWidget(self.ui.dredging_group, dredging_table.to_dataframe())
            self.dredging_table.setObjectName('dredging_table')
            self.dredging_table.setMinimumHeight(205)  # min height from the XML
            self.ui.dredging_group.layout().insertWidget(0, self.dredging_table)

            table_id = int(self.struct_data['bypass_table'].item())
            if table_id <= 0:
                table_id = self.comp_id
                self.struct_data['bypass_table'] = table_id
            bypass_table = self.all_data.bypass_table_from_id(table_id)
            self.bypass_table = BypassTableWidget(self.ui.bypass_group, bypass_table.to_dataframe())
            self.bypass_table.setObjectName('bypass_table')
            self.bypass_table.setMinimumHeight(205)  # min height from the XML
            self.ui.bypass_group.layout().insertWidget(0, self.bypass_table)

            table_id = int(self.struct_data['beach_fill_table'].item())
            if table_id <= 0:
                table_id = self.comp_id
                self.struct_data['beach_fill_table'] = table_id
            beach_fill_table = self.all_data.beach_fill_table_from_id(table_id)
            self.beach_fill_table = BeachFillTableWidget(self.ui.beach_fill_group, beach_fill_table.to_dataframe())
            self.beach_fill_table.setObjectName('beach_fill_table')
            self.beach_fill_table.setMinimumHeight(205)  # min height from the XML
            self.ui.beach_fill_group.layout().insertWidget(0, self.beach_fill_table)

    def _connect_widgets(self):
        """Sets up the signal/slot connections for the dialog."""
        self.ui.buttonBox.accepted.connect(self.accept)
        self.ui.buttonBox.rejected.connect(self.reject)
        self.ui.buttonBox.helpRequested.connect(self.help_requested)
        self.ui.struct_type_cbx.currentIndexChanged[int].connect(self._set_structure_stack)
        self.ui.transmission_type_cbx.currentIndexChanged[int].connect(self._set_transmission_type)

    def _setup_combo_boxes(self):
        """Sets up the options for the combo-boxes."""
        self.ui.struct_type_cbx.addItems(STRUCT_TYPES)
        self.ui.transmission_type_cbx.addItems(TRANSMISSION_TYPES)

    def _setup_dredging_table_combo(self):
        """Sets up the options for the combo box located in the Dredging table."""
        # Create combo box for Dredging Table containing 'shoal_mine_types'
        pass

    def _setup_edit_fields(self):
        """Sets up the validators on edit fields."""
        self._add_breakwater_validators()
        self._add_groin_validators()
        self._add_inlet_validators()

    def _add_breakwater_validators(self):
        """Sets up the validators on edit fields for breakwaters."""
        self.ui.depth1.setValidator(self.nonzero_dbl_validator)
        self.ui.depth2.setValidator(self.nonzero_dbl_validator)
        self.ui.transmission_const.setValidator(self.dbl_validator)
        self.ui.height.setValidator(self.dbl_validator)
        self.ui.width.setValidator(self.dbl_validator)
        self.ui.seaward_slope.setValidator(self.dbl_validator)
        self.ui.shoreward_slope.setValidator(self.dbl_validator)
        self.ui.armor_d50.setValidator(self.dbl_validator)
        self.ui.bw_permeability.setValidator(self.dbl_validator)

    def _add_groin_validators(self):
        """Sets up the validator on edit fields for groins."""
        self.ui.groin_permeability.setValidator(self.dbl_validator)
        self.ui.seaward_depth.setValidator(self.nonzero_dbl_validator)

    def _add_inlet_validators(self):
        """Sets up the validators on edit fields for inlets."""
        self.ui.ebb_shoal_init.setValidator(self.dbl_validator)
        self.ui.ebb_shoal_equil.setValidator(self.dbl_validator)
        self.ui.flood_shoal_init.setValidator(self.dbl_validator)
        self.ui.flood_shoal_equil.setValidator(self.dbl_validator)
        self.ui.left_bypass_init.setValidator(self.dbl_validator)
        self.ui.left_bypass_equil.setValidator(self.dbl_validator)
        self.ui.left_bypass_coeff.setValidator(self.dbl_validator)
        self.ui.right_bypass_init.setValidator(self.dbl_validator)
        self.ui.right_bypass_equil.setValidator(self.dbl_validator)
        self.ui.right_bypass_coeff.setValidator(self.dbl_validator)
        self.ui.left_attach_init.setValidator(self.dbl_validator)
        self.ui.left_attach_equil.setValidator(self.dbl_validator)
        self.ui.right_attach_init.setValidator(self.dbl_validator)
        self.ui.right_attach_equil.setValidator(self.dbl_validator)

    def _load_data(self):
        """Loads the data from self.struct_data into the appropriate widgets."""
        # In case a previous project has the string length bug (type gets cropped to 7 chars),
        # find the matching index based on it being a substring of the correct type.
        try:
            self.ui.struct_type_cbx.setCurrentIndex(find_matching_index(
                STRUCT_TYPES, self.struct_data['struct_type_cbx'].item()))
        except Exception:
            raise RuntimeError(f'Could not find "{self.struct_data["struct_type_cbx"].item()} in available types."')

        self.ui.depth1.setText(str(self.struct_data['depth1'].item()))
        self.ui.depth2.setText(str(self.struct_data['depth2'].item()))
        self.ui.transmission_type_cbx.setCurrentIndex(TRANSMISSION_TYPES.index(
            self.struct_data['transmission_type_cbx'].item()))
        self.ui.transmission_const.setText(str(self.struct_data['transmission_const'].item()))
        if 'height' not in self.struct_data:
            self.struct_data['height'] = self.struct_data['freeboard_msl']
        self.ui.height.setText(str(self.struct_data['height'].item()))
        self.ui.width.setText(str(self.struct_data['width'].item()))
        self.ui.seaward_slope.setText(str(self.struct_data['seaward_slope'].item()))
        self.ui.shoreward_slope.setText(str(self.struct_data['shoreward_slope'].item()))
        self.ui.armor_d50.setText(str(self.struct_data['armor_d50'].item()))
        self.ui.bw_permeability.setText(str(self.struct_data['bw_permeability'].item()))
        self.ui.groin_permeability.setText(str(self.struct_data['groin_permeability'].item()))
        self.ui.diffracting_group.setChecked(self.struct_data['diffracting_chk'].item() != 0)
        self.ui.seaward_depth.setText(str(self.struct_data['seaward_depth'].item()))
        self.ui.name.setText(TextConverter.get_text_for_field(self.struct_data['name'].item()))
        self.ui.ebb_shoal_init.setText(str(self.struct_data['ebb_shoal_init'].item()))
        self.ui.ebb_shoal_equil.setText(str(self.struct_data['ebb_shoal_equil'].item()))
        self.ui.flood_shoal_init.setText(str(self.struct_data['flood_shoal_init'].item()))
        self.ui.flood_shoal_equil.setText(str(self.struct_data['flood_shoal_equil'].item()))
        self.ui.left_bypass_init.setText(str(self.struct_data['left_bypass_init'].item()))
        self.ui.left_bypass_equil.setText(str(self.struct_data['left_bypass_equil'].item()))
        self.ui.left_bypass_coeff.setText(str(self.struct_data['left_bypass_coeff'].item()))
        self.ui.right_bypass_init.setText(str(self.struct_data['right_bypass_init'].item()))
        self.ui.right_bypass_equil.setText(str(self.struct_data['right_bypass_equil'].item()))
        self.ui.right_bypass_coeff.setText(str(self.struct_data['right_bypass_coeff'].item()))
        self.ui.left_attach_init.setText(str(self.struct_data['left_attach_init'].item()))
        self.ui.left_attach_equil.setText(str(self.struct_data['left_attach_equil'].item()))
        self.ui.right_attach_init.setText(str(self.struct_data['right_attach_init'].item()))
        self.ui.right_attach_equil.setText(str(self.struct_data['right_attach_equil'].item()))
        #
        self.ui.dredging_group.setChecked(self.struct_data['use_dredging_table'].item() != 0)
        self.ui.bypass_group.setChecked(self.struct_data['use_bypass_table'].item() != 0)
        self.ui.beach_fill_group.setChecked(self.struct_data['use_beach_fill_table'].item() != 0)
        #
        idx = self.ui.struct_type_cbx.currentIndex()
        self._set_structure_stack(idx)  # Stack indices are not 1:1 with structure types
        idx = self.ui.transmission_type_cbx.currentIndex()
        self._set_transmission_type(idx)

    def _set_structure_stack(self, idx):
        """Set structure stack to the correct page depending on index.

        Args:
            idx (:obj:`int`): Index value of the combo box calling this function.
        """
        if idx in [0, 1, 2, 4, 11, 12, 13, 14]:
            self.ui.structure_stack.setCurrentIndex(0)  # No options
        elif idx == 3:
            self.ui.structure_stack.setCurrentIndex(1)  # Breakwater
        elif idx in [5, 7, 8]:
            self.ui.structure_stack.setCurrentIndex(2)  # Groin/Jetty
        elif idx == 6:
            self.ui.structure_stack.setCurrentIndex(3)  # Inlet
        elif idx == 9:
            self.ui.structure_stack.setCurrentIndex(4)  # Bypass Event
        elif idx == 10:
            self.ui.structure_stack.setCurrentIndex(5)  # Beach Fill Event

    def _set_transmission_type(self, idx):
        """Called when the transmission type combobox option changes.

        Args:
            idx (:obj:`int`): Index value of the combo box calling this function.
        """
        is_const = idx == 0
        is_dangremond = idx == 3
        is_other = idx in [1, 2]
        self.ui.transmission_const.setVisible(is_const)
        self.ui.transmission_const_label.setVisible(is_const)
        self.ui.height_label.setVisible(not is_const)
        self.ui.width_label.setVisible(not is_const)
        self.ui.seaward_slope_label.setVisible(not is_const)
        self.ui.shoreward_slope_label.setVisible(not is_const)
        self.ui.height.setVisible(not is_const)
        self.ui.width.setVisible(not is_const)
        self.ui.seaward_slope.setVisible(not is_const)
        self.ui.shoreward_slope.setVisible(not is_const)
        self.ui.armor_d50_label.setVisible(is_other)
        self.ui.armor_d50.setVisible(is_other)
        self.ui.bw_permeability_label.setVisible(is_dangremond)
        self.ui.bw_permeability.setVisible(is_dangremond)

    def _save_struct_data(self):
        """Save data from the structure arc attributes."""
        self.struct_data['struct_type_cbx'] = self.ui.struct_type_cbx.currentText()
        self.struct_data['depth1'] = float(self.ui.depth1.text())
        self.struct_data['depth2'] = float(self.ui.depth2.text())
        self.struct_data['transmission_type_cbx'] = self.ui.transmission_type_cbx.currentText()
        self.struct_data['transmission_const'] = float(self.ui.transmission_const.text())
        self.struct_data['height'] = float(self.ui.height.text())
        self.struct_data['width'] = float(self.ui.width.text())
        self.struct_data['seaward_slope'] = float(self.ui.seaward_slope.text())
        self.struct_data['shoreward_slope'] = float(self.ui.shoreward_slope.text())
        self.struct_data['armor_d50'] = float(self.ui.armor_d50.text())
        self.struct_data['bw_permeability'] = float(self.ui.bw_permeability.text())
        self.struct_data['groin_permeability'] = float(self.ui.groin_permeability.text())
        self.struct_data['diffracting_chk'] = 1 if self.ui.diffracting_group.isChecked() else 0
        self.struct_data['seaward_depth'] = float(self.ui.seaward_depth.text())
        self.struct_data['name'] = TextConverter.get_text_from_field(self.ui.name.text())
        self.struct_data['ebb_shoal_init'] = float(self.ui.ebb_shoal_init.text())
        self.struct_data['ebb_shoal_equil'] = float(self.ui.ebb_shoal_equil.text())
        self.struct_data['flood_shoal_init'] = float(self.ui.flood_shoal_init.text())
        self.struct_data['flood_shoal_equil'] = float(self.ui.flood_shoal_equil.text())
        self.struct_data['left_bypass_init'] = float(self.ui.left_bypass_init.text())
        self.struct_data['left_bypass_equil'] = float(self.ui.left_bypass_equil.text())
        self.struct_data['left_bypass_coeff'] = float(self.ui.left_bypass_coeff.text())
        self.struct_data['right_bypass_init'] = float(self.ui.right_bypass_init.text())
        self.struct_data['right_bypass_equil'] = float(self.ui.right_bypass_equil.text())
        self.struct_data['right_bypass_coeff'] = float(self.ui.right_bypass_coeff.text())
        self.struct_data['left_attach_init'] = float(self.ui.left_attach_init.text())
        self.struct_data['left_attach_equil'] = float(self.ui.left_attach_equil.text())
        self.struct_data['right_attach_init'] = float(self.ui.right_attach_init.text())
        self.struct_data['right_attach_equil'] = float(self.ui.right_attach_equil.text())
        self.struct_data['use_dredging_table'] = 1 if self.ui.dredging_group.isChecked() else 0
        self.all_data.set_dredging_table(int(self.struct_data['dredging_table'].item()),
                                         self.dredging_table.model.data_frame.to_xarray())
        self.struct_data['use_bypass_table'] = 1 if self.ui.bypass_group.isChecked() else 0
        self.all_data.set_bypass_table(int(self.struct_data['bypass_table'].item()),
                                       self.bypass_table.model.data_frame.to_xarray())
        self.struct_data['use_beach_fill_table'] = 1 if self.ui.beach_fill_group.isChecked() else 0
        self.all_data.set_beach_fill_table(int(self.struct_data['beach_fill_table'].item()),
                                           self.beach_fill_table.model.data_frame.to_xarray())

    def _get_selected_file(self, widget):
        """Gets the selected file text.

        Args:
            widget (:obj:`QLabel`): The label widget that shows the file path.

        Returns:
            Returns the path of the selected file. The file path is an empty string if nothing is selected.
        """
        path = widget.text()
        if path == '(none_selected)':
            path = ''
        elif os.path.isfile(path):
            proj_dir = self.all_data.info.attrs['proj_dir']
            if os.path.isdir(proj_dir):
                path = try_to_compute_relative_path(proj_dir, path)
        return path

    def _set_selected_file_for_widget(self, widget, key):
        """Sets the selected file text for the label widget.

        Args:
            widget (:obj:`QLabel`): The label widget that shows the file path.
            key (:obj:`str`): The key in self.bc_data for the widget.
        """
        path = self.bc_data[key].item()
        if path:
            widget.setText(path)

    def _set_selected_dataset_path_for_widget(self, widget, key):
        """Sets the selected dataset path for the label widget.

        Args:
            widget (:obj:`QLabel`): The label widget that shows the dataset path.
            key (:obj:`str`): The key in self.bc_data for the widget.
        """
        dset_uuid = self.bc_data[key].item()
        try:
            _ = uuid.UUID(dset_uuid, version=4)  # throw an exception if an invalid UUID
            dset_path = build_tree_path(self.pe_tree, dset_uuid)
        except ValueError:
            dset_path = '(none selected)'
        widget.setText(dset_path if dset_path else '(none selected)')

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._save_struct_data()
        super().accept()
