"""Module for DatasetSelectorMixin."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util, TreeNode

# 4. Local modules
from xms.gmi.data.generic_model import Parameter
from xms.gmi.gui.dataset_callback import DatasetRequest


class DatasetSelectorMixin:
    """A Dynamic Model Interface (DMI) component base for GMI-based coverage components."""
    def __init__(self, main_file: Optional[str | Path]):
        """
        Initialize the component.

        Args:
            main_file: The component's main-file.
        """
        super().__init__(main_file)
        self._project_tree: Optional[TreeNode] = None
        self._query: Optional[Query] = None

    def _dataset_callback(self, request: DatasetRequest, parameter: Parameter) -> str | TreeNode:
        """
        Handle a request for information when picking a dataset.

        This should match `xms.gmi.gui.dataset_callback.DatasetCallback`.

        Notes:
            - `self._query` will be initialized and available before this method is called.
            - If `request` is `DatasetRequest.GetTree`, the returned node will be a clean copy of the entire project
              tree. Derived classes can safely mutate it for filtering purposes.

        Args:
            request: The requested operation.
            parameter: The parameter the request is for. If `request` is `DatasetRequest.GetLabel`, then
                `parameter.value` will be initialized to the UUID of the dataset to get the label for.

        Returns:
            - If `request` is `DatasetRequest.GetLabel`, returns a label to identify the dataset with to the user.
            - If `request` is `DatasetRequest.GetTree`, returns a tree for picking a new dataset for the parameter.
        """
        if self._project_tree is None:
            self._project_tree = self._query.copy_project_tree()
        if request == DatasetRequest.GetLabel:
            return tree_util.build_tree_path(self._project_tree, parameter.value)
        if request == DatasetRequest.GetTree:
            return self._project_tree
        raise AssertionError('Unknown request.')  # pragma: nocover
