"""SimComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
from typing import Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query, XmsEnvironment as XmEnv
from xms.api.tree import tree_util

# 4. Local modules
from xms.gmi.components.coverage_component import CoverageComponent
from xms.gmi.components.gmi_component import GmiComponent
from xms.gmi.components.material_component import MaterialComponent
from xms.gmi.data.generic_model import GenericModel
from xms.gmi.data.sim_data import SimData
from xms.gmi.gui.group_set_dialog import GroupSetDialog


class SimComponent(GmiComponent):
    """
    A hidden Dynamic Model Interface (DMI) component for the GMI model simulation.

    This component handles events that happen to the simulation tree item in
    the project explorer.
    """
    def __init__(self, main_file: Path | str, generic_model: Optional[GenericModel] = None):
        """
        Initialize the component class.

        Args:
            main_file: The main file associated with this component.
            generic_model: The simulation's model. If `None`, uses a default one.
        """
        super().__init__(main_file, generic_model)
        self.tree_commands = [
            # (menu_text, method_on_self_to_call_when_clicked)
            ('Model Control...', 'open_model_control'),
        ]

    def _get_data_with_model(self, model: GenericModel):
        """
        Get a data manager for this component.

        Derived classes can override this to provide their own data manager. Implementations will typically look
        something like

        ```
        def _get_data(self):
            return SomeDataManager(self.main_file, model)
        ```

        GMI data classes define this because they have to be ready to deal with arbitrary models, but derived models
        generally won't need this because they only have a single model. They will generally prefer to define
        `self._get_data` instead. That alternative is the same as this, except it has no `model` parameter.
        """
        return SimData(self.main_file, model)

    def link_event(self, link_dict: dict[str, list[str]],
                   lock_state: bool) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        This will be called when one or more coverages, ugrids, or other components are linked to this component.

        Args:
            link_dict: A dictionary mapping UUIDs of objects being linked -> list of "takes". The "takes" will always
                be a single-element list since GMI only allows one component per coverage. The actual value is defined
                in the XML by the `<declare_parameter>` tags and is used to figure out what type of coverage is being
                linked.
            lock_state: Whether the component is locked for editing. Do not change the files if locked.

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        request = ActionRequest(
            main_file=self.main_file,
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='check_link_compatibility',
            comp_uuid=self.uuid,
            parameters=link_dict,
        )

        return [], [request]

    def check_link_compatibility(
        self, query: Query, params: list[dict[str, list[str]]]
    ) -> tuple[list[tuple[str, str]], list[ActionRequest]]:
        """
        Check whether a recently linked item is compatible with the simulation.

        If incompatible, removes the link and reports an error to the user.

        Args:
            query: Query to interact with XMS.
            params: List of dictionaries mapping coverage UUID to coverage type.

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        params = params[0]
        simulation_model = self.data.generic_model
        sim_uuid = query.parent_item_uuid()
        messages = []
        tree = None

        for uuid in params:
            if params[uuid][0] == 'mesh':
                continue  # Meshes are always allowed
            is_boundary = params[uuid][0] == 'boundary_conditions'
            name = 'CoverageComponent' if is_boundary else 'MaterialComponent'
            do_component = query.item_with_uuid(uuid, model_name='GMI', unique_name=name)
            main_file = do_component.main_file
            component = CoverageComponent(main_file) if is_boundary else MaterialComponent(main_file)
            component_model: GenericModel = component.data.generic_model
            if not simulation_model.compatible(component_model):
                tree = tree or query.project_tree
                query.unlink_item(sim_uuid, uuid)
                coverage_name = tree_util.find_tree_node_by_uuid(tree, uuid).name
                simulation_name = tree_util.find_tree_node_by_uuid(tree, sim_uuid).name
                messages.append(('ERROR', f"'{coverage_name}' is not compatible with this simulation."))

                directory = Path(XmEnv.xms_environ_temp_directory()) / 'gmi-rejected-link-info'
                os.makedirs(directory, exist_ok=True)
                with open(directory / f'{simulation_name} - {sim_uuid}.txt', 'w') as f:
                    f.write(simulation_model.to_pretty_string())
                with open(directory / f'{coverage_name} - {uuid}.txt', 'w') as f:
                    f.write(component_model.to_pretty_string())

        return messages, []

    def open_model_control(self, query: Query, params: list[dict], win_cont: QWidget):
        """
        Open the Model Control dialog and save component data state on OK.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            win_cont: The window container.

        Returns:
            A tuple of (messages, action_requests).

            - messages: List of tuples with the first element of the tuple being the message
              level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
            - action_requests: List of actions for XMS to perform.
        """
        self._query = query

        generic_model = self.data.generic_model
        global_values = self.data.global_values
        generic_model.global_parameters.restore_values(global_values)

        dlg = GroupSetDialog(
            parent=win_cont,
            section=generic_model.global_parameters,
            get_curve=self.data.get_curve,
            add_curve=self.data.add_curve,
            is_interior=False,
            dlg_name='xms.gmi.components.sim_component',
            enable_unchecked_groups=True,
            dataset_callback=self._dataset_callback,
        )

        if dlg.exec():
            # Update the attribute datasets
            values = dlg.section.extract_values()
            self.data.global_values = values
            self.data.commit()
        return [], []
