"""MaterialData class."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.gmi.data.coverage_data import CoverageData
from xms.gmi.data.generic_model import GenericModel


class MaterialData(CoverageData):
    """
    Manages data file for a material component.

    See `CoverageData` for the majority of `MaterialData` functionality.

    `MaterialData` derives from `CoverageData` to add material value storage and retrieval. It does not define its own
     datasets; it uses the polygon one instead.
    """
    def __init__(self, data_file: Optional[str | Path] = None, generic_model: GenericModel | None = None):
        """
        Initialize the data class.

        Args:
            data_file: The netcdf file (with path) associated with this instance data. Probably the owning
                       component's main file. If not provided, a default one will be chosen.
            generic_model: The model this coverage should use.
        """
        super().__init__(data_file, generic_model)

    @property
    def _main_file_name(self):
        """What to name the component's main file."""
        return 'gmi_materials.nc'

    @property
    def material_values(self) -> str:
        """
        The shared values for material parameters.

        This does not include group activity.
        """
        if 'MATERIAL_VALUES' in self.info.attrs:
            return self.info.attrs['MATERIAL_VALUES']
        # Let GenericModel define the defaults in case it wants to change them.
        return GenericModel().material_parameters.extract_values()

    @material_values.setter
    def material_values(self, value: str):
        """
        The shared values for material parameters.

        This does not include group activity.
        """
        self.info.attrs['MATERIAL_VALUES'] = value
