"""Module for the XmsData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"
__all__ = ['MISSING', 'XmsData']

# 1. Standard Python modules
from functools import cached_property
from typing import Callable, Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util, TreeNode
from xms.constraint import Grid, read_grid_from_file
from xms.data_objects.parameters import Coverage, Projection, UGrid as DoUGrid
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.gmi.component_bases.coverage_component_base import CoverageComponentBase
from xms.gmi.data.generic_model import GenericModel, Section
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData
from xms.gmi.data_bases.sim_base_data import SimBaseData

MISSING = object()


class FakeComponent(CoverageComponentBase):
    def _get_section(self, target: TargetType) -> Section:
        return GenericModel().section_from_target_type(target)


class XmsData:
    """Class for wrapping `Query` with a higher-level interface and caching."""
    model_name = ''
    sim_component_unique_name = ''
    linkable_ugrid_types: list[str] = ['TI_UGRID_PTR', 'TI_CGRID2D_PTR']
    get_model: Callable[[], GenericModel] = None

    # PyCharm doesn't know that cached properties can be set.
    # noinspection PyPropertyAccess
    def __init__(
        self,
        query: Optional[Query] = None,
        model_control: Optional[Section] = None,
        linked_grid: Optional[Grid] = None,
        linked_grid_projection: Optional[Projection] = None,
        linked_coverages_and_components: Optional[dict[str, tuple[Coverage, CoverageBaseData]]] = None,
    ):
        """
        Initialize the class.

        When running in XMS, this should be initialized like `XmsData(query)` with a valid, initialized Query. When
        running under tests, `query` should be None and any other necessary parameters should be filled in instead.

        Most parameters override the attribute with the same name, forcing it to return the provided value instead of
        whatever `query` would have returned. When an overridden value is provided, `query` will not be consulted for
        it. This allows tests to inject their own values without a `Query`.

        Some items may be passed the MISSING object above to indicate the item should skip Query and just return None.

        Args:
            query: Interprocess communication object.
            model_control: Overrides self.model_control.
            linked_grid: Overrides self.linked_grid. May be MISSING.
            linked_grid_projection: Overrides self.linked_grid_projection. May be MISSING.
            linked_coverages_and_components: Mapping from coverage class name to tuple of coverage and data manager.
                If passed and a class name is present in it, then Query will not be consulted when retrieving that item.
                Query will still be consulted for missing items, even if this is passed, so if it's desired that None
                be returned, then the value for the key should be set to (None, None).
        """
        self._query = query
        self.unlinks = []

        if model_control is not None:
            self.model_control = model_control

        if linked_grid is MISSING:
            self.linked_grid = None
        elif linked_grid is not None:
            self.linked_grid = linked_grid

        if linked_grid_projection is MISSING:
            self.linked_grid_projection = None
        elif linked_grid_projection is not None:
            self.linked_grid_projection = linked_grid_projection

        self._linked_coverages_and_components = linked_coverages_and_components or {}

    @cached_property
    def _sim_node(self) -> TreeNode:
        tree = self._query.copy_project_tree()
        possible_sim_uuid = self._query.current_item_uuid()
        sim_node = tree_util.find_tree_node_by_uuid(tree, possible_sim_uuid)
        if not sim_node:
            possible_sim_uuid = self._query.parent_item_uuid()
            sim_node = tree_util.find_tree_node_by_uuid(tree, possible_sim_uuid)
        return sim_node

    @cached_property
    def model_control(self) -> Section:
        """The main-file for the current simulation."""
        assert self.model_name and self.sim_component_unique_name and self.get_model
        sim_uuid = self._sim_node.uuid
        sim_comp = self._query.item_with_uuid(
            sim_uuid, model_name=self.model_name, unique_name=self.sim_component_unique_name
        )
        sim_data = SimBaseData(sim_comp.main_file)
        section = type(self).get_model().global_parameters
        section.restore_values(sim_data.global_values)
        return section

    def _linked_coverage_and_component(
        self, component_class_name: str
    ) -> tuple[Optional[Coverage], Optional[CoverageBaseData]]:
        """
        Get a coverage and component that are linked to the simulation.

        Args:
            component_class_name: Class name of the component for the coverage to retrieve.

        Returns:
            Tuple of (coverage, data), where Coverage is the geometry and data is the coverage's data manager. The
            data manager will have its component_id_map initialized.
        """
        assert self.model_name

        if component_class_name not in self._linked_coverages_and_components:
            sim_node = self._sim_node
            coverage_node = tree_util.descendants_of_type(
                sim_node,
                xms_types=['TI_COVER_PTR'],
                recurse=True,
                allow_pointers=True,
                only_first=True,
                coverage_type=component_class_name
            )

            if not coverage_node:
                return None, None

            coverage = self._query.item_with_uuid(coverage_node.uuid)
            do_comp = self._query.item_with_uuid(
                coverage_node.uuid, model_name=self.model_name, unique_name=component_class_name
            )
            if not coverage or not do_comp or not do_comp.main_file:
                raise AssertionError('Unexpected coverage type')  # pragma: nocover  Developer mistake.

            component = FakeComponent(do_comp.main_file)
            self._query.load_component_ids(component, points=True, arcs=True, polygons=True)
            self._linked_coverages_and_components[component_class_name] = (coverage, component.data)

        return self._linked_coverages_and_components[component_class_name]

    @cached_property
    def linked_grid(self) -> Optional[Grid]:
        """
        The grid that is linked to the simulation.

        Will be None if no grid is linked.
        """
        ugrid, _projection = self._grid_and_projection
        return ugrid

    @cached_property
    def linked_grid_projection(self) -> Optional[Projection]:
        """
        The projection of the UGrid that is linked to the simulation.

        Will be None if no UGrid is linked.
        """
        _ugrid, projection = self._grid_and_projection
        return projection

    @cached_property
    def _grid_and_projection(self) -> tuple[Optional[Grid], Optional[Projection]]:
        """Helper to get the UGrid linked to the simulation and its projection."""
        ugrid_item: TreeNode = tree_util.descendants_of_type(
            self._sim_node, xms_types=self.linkable_ugrid_types, recurse=False, allow_pointers=True, only_first=True
        )
        if not ugrid_item:
            return None, None

        do_ugrid: DoUGrid = self._query.item_with_uuid(ugrid_item.uuid)
        co_grid: Grid = read_grid_from_file(do_ugrid.cogrid_file)
        return co_grid, do_ugrid.projection

    def unlink_grid(self):
        """Unlink the currently linked grid from the simulation."""
        grid = self.linked_grid
        grid_uuid = grid.uuid
        if self._query:
            self._query.unlink_item(self._sim_node.uuid, grid_uuid)
        else:
            self.unlinks.append(grid_uuid)
