"""Utility functions to deal with query."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import logging
from pathlib import Path
import uuid

# 2. Third party modules
from PySide2.QtWidgets import QWidget  # Needed for the type hints. Don't delete it.

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv
from xms.api.tree import tree_util, TreeNode
from xms.constraint import Grid, read_grid_from_file
from xms.core.filesystem import filesystem
from xms.data_objects.parameters import Component, Coverage
from xms.datasets.dataset_reader import DatasetReader  # Needed for the type hints. Don't delete it.
from xms.gmi.data.generic_model import Group
from xms.guipy import file_io_util
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.gssha.components import gmi_util
from xms.gssha.components.bc_coverage_component import BcCoverageComponent
from xms.gssha.file_io import io_util


def find_bc_coverage_and_component(sim_node: TreeNode, query: Query) -> tuple[Coverage, BcCoverageComponent]:
    """Finds and stores the linked coverage."""
    coverage_ptr_nodes = io_util.get_coverage_pointers(sim_node, '')
    if not coverage_ptr_nodes:
        return None, None

    bc_cov = query.item_with_uuid(coverage_ptr_nodes[0].uuid)
    bc_comp = get_bc_coverage_component(coverage_ptr_nodes[0].uuid, query)
    query.load_component_ids(bc_comp, points=True, arcs=True, polygons=True)
    return bc_cov, bc_comp


def build_solution_component(sol_file_path: Path, sim_uuid: str, unique_name: str, locked: bool) -> Component:
    """Create a component for solution files.

    Args:
        sol_file_path: Filesystem path to the solution file.
        sim_uuid: The UUID of the simulation that this solution belongs to.
        unique_name: XML definition unique_name of the component to build.
        locked: True if component is to be locked.

    Returns:
        The data_objects component to send back to XMS for the new component.
    """
    # Create main file path and create the actual folder
    comp_uuid = str(uuid.uuid4())
    main_file_path = Path(XmEnv.xms_environ_temp_directory()) / 'Components' / comp_uuid / f'solution.{unique_name}'
    main_file_path.parent.mkdir(parents=True, exist_ok=True)

    # Compute the relative path from the XMS project to the solution file
    xms_project_file_path = Path(XmEnv.xms_environ_project_path())
    rel_path = filesystem.compute_relative_path(xms_project_file_path.parent, sol_file_path)

    # We store both the full path and the relative path from the component uuid dir because there are multiple
    # scenarios:
    # 1. Entire project might be moved to a new computer/location. Full path won't work, only relative will
    # 2. In save_to_location, the XMS project might be saved to a new location / new name. We need the full path to
    # compute the new relative path from the new XMS project to the old solution file.
    cards = {'SIM_UUID': sim_uuid, 'SOLUTION_FILE_FULL': str(sol_file_path), 'SOLUTION_FILE_RELATIVE': str(rel_path)}
    file_io_util.write_json_file(cards, str(main_file_path))
    do_comp = Component(
        name=sol_file_path.name,
        comp_uuid=comp_uuid,
        main_file=str(main_file_path),
        model_name='GSSHA',
        unique_name=unique_name,
        locked=locked
    )
    return do_comp


def find_solution_file(solution_file_full: str, solution_file_relative: str) -> Path | None:
    """Returns the location of the solution file or '' if not found.

    We store both the full path and the relative path from the component uuid dir because there are multiple
    scenarios:
    1. Entire project might be moved to a new computer/location. Full path won't work, only relative will
    2. In save_to_location, we need the full path to compute the new relative path

    Args:
        solution_file_full: Full path to the solution file.
        solution_file_relative: Relative path from XMS project file to the solution file.

    Returns:
        (Path): See description.
    """
    # Use the full path if it's valid
    if solution_file_full and Path(solution_file_full).is_file():
        return Path(solution_file_full)

    # Use the relative path if it's valid
    xms_project_file_path = Path(XmEnv.xms_environ_project_path())
    full_path = filesystem.resolve_relative_path(xms_project_file_path.parent, solution_file_relative)
    if full_path and Path(full_path).is_file():
        return Path(full_path)

    return None  # We can't find it


def get_bc_coverage_component(cov_uuid: str, query: Query):
    """Returns the bc coverage component object.

    Args:
        cov_uuid: The Coverage or tree node UUID.
        query: Object for communicating with XMS

    Returns:
        See description.
    """
    from xms.gssha.components.bc_coverage_component import BcCoverageComponent  # here to avoid circular imports
    return get_coverage_component(cov_uuid, BcCoverageComponent, 'BcCoverageComponent', query)


def get_coverage_component(cov_uuid: str, component_class, unique_name: str, query: Query):
    """Returns the coverage component object.

    Args:
        cov_uuid: Coverage or tree node uuid.
        component_class: Component class.
        unique_name: Component unique_name from xml
        query: Object for communicating with XMS

    Returns:
        See description.
    """
    do_comp = query.item_with_uuid(cov_uuid, unique_name=unique_name, model_name='GSSHA')
    if not do_comp:
        return None
    component = component_class(do_comp.main_file)
    return component


def get_dataset(group: Group, parameter: str, query: Query) -> tuple['DatasetReader | None', str | None]:
    """Returns the dataset and dataset name (or None if it can't find it).

    Args:
        group: The generic model group.
        parameter: The generic model parameter name.
        query (Query): Object for communicating with XMS

    Returns:
        See description.
    """
    dataset_uuid = group.parameter(parameter).value
    if dataset_uuid:
        dataset = query.item_with_uuid(dataset_uuid)
        if dataset:
            dataset_node = tree_util.find_tree_node_by_uuid(query.project_tree, dataset_uuid)
            return dataset, dataset_node.name

    return None, None


def get_sim_name(query) -> str:
    """Returns the name of the simulation.

    Args:
        query (xmsapi.dmi.Query): Object for communicating with GMS

    Returns:
        (str): See description.
    """
    sim_uuid = query.parent_item_uuid()
    return tree_util.find_tree_node_by_uuid(query.project_tree, sim_uuid).name


def get_gssha_file_path(query) -> Path:
    """Returns the location of the saved simulation (.gssha file), and the simulation tree node.

    Args:
        query (xmsapi.dmi.Query): Object for communicating with GMS

    Returns:
        (Path): See description.
    """
    project_file = Path(XmEnv.xms_environ_project_path())
    project_dir = project_file.parent
    project_base = project_file.stem
    sim_name = get_sim_name(query)
    grok_file_path = project_dir / f'{project_base}_models' / 'GSSHA' / sim_name / f'{sim_name}.gssha'
    return grok_file_path


def get_default_bc_coverage_node(project_tree: TreeNode) -> 'TreeNode | None':
    """Returns the GSSHA bc coverage - the one linked to the active sim, or the first one found, or None."""
    if not project_tree:
        return None

    cov_type = 'Boundary Conditions'
    bc_nodes = tree_util.descendants_of_type(project_tree, coverage_type=cov_type, model_name='GSSHA')
    return bc_nodes[0] if bc_nodes else None


def get_active_sim_node(project_tree: TreeNode) -> 'TreeNode | None':
    """Returns the active GSSHA sim tree node, or None."""
    sim_nodes = tree_util.descendants_of_type(project_tree, xms_types=['TI_DYN_SIM'], model_name='GSSHA')
    active_sim_node = None
    for sim_node in sim_nodes:
        if sim_node.is_active_item:
            active_sim_node = sim_node
            break
    return active_sim_node


def read_co_grid(query: Query, sim_node: TreeNode) -> Grid:
    """Finds, reads, and returns the grid, or raises a RuntimeError if there is no linked UGrid.

    Args:
        query: Object for communicating with XMS
        sim_node: The simulation tree node.

    Returns:
        The Grid.
    """
    ugrid_node = get_ugrid_node_or_warn(query.project_tree, None, sim_node=sim_node)
    if not ugrid_node:
        raise RuntimeError('No linked UGrid.')
    do_ugrid = query.item_with_uuid(ugrid_node.uuid)
    file = do_ugrid.cogrid_file
    return read_grid_from_file(file)


def get_ugrid_node_or_warn(
    project_tree: TreeNode,
    win_cont: 'QWidget | None',
    sim_node: 'TreeNode | None' = None,
    query: 'Query | None' = None
) -> 'TreeNode | None':
    """Gets the tree node of the linked UGrid, or, if none, warns the user and returns None.

    Args:
        project_tree: The tree.
        win_cont: The window container, so we can raise a warning if UGrid not found. If None, no warning will be shown.
        sim_node: The simulation tree node.
        query: Object for communicating with XMS

    Returns:
        See description.
    """
    if not sim_node and not query:
        raise ValueError('Must pass sim_node or query.')

    if not sim_node and query:
        sim_node = tree_util.find_tree_node_by_uuid(project_tree, query.parent_item_uuid())
    ugrid_ptr_node = tree_util.descendants_of_type(
        sim_node, allow_pointers=True, xms_types=['TI_UGRID_PTR'], only_first=True
    )
    if not ugrid_ptr_node:
        message = 'A UGrid needs to be linked to the simulation first.'
        if win_cont:
            message_box.message_with_ok(parent=win_cont, message=message, app_name='SMS')
        else:
            log = logging.getLogger('xms.gssha')
            log.warning(message)
        return None
    return tree_util.find_tree_node_by_uuid(project_tree, ugrid_ptr_node.uuid)


def create_query() -> Query:
    """Creates and returns a Query.

    This exists so it can be mocked in tests.
    """
    return Query(timeout=300000)  # pragma no cover - can't test this


def do_comp_from_bc_comp(bc_comp: BcCoverageComponent) -> Component:
    """Creates and returns a data_objects Component for the BcCoverageComponent.

    Args:
        bc_comp: The BcCoverageComponent

    Returns:
        See description.
    """
    return gmi_util.make_do_cov_comp(bc_comp.main_file, 'BcCoverageComponent', 'GSSHA', bc_comp.uuid)
