"""OutletHydrographComponent class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
import pandas as pd
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.guipy import file_io_util
# from xms.guipy.dialogs import message_box
from xms.guipy.dialogs.xy_series_editor import XySeriesEditor

# 4. Local modules
from xms.gssha.components import dmi_util
from xms.gssha.components.gssha_component_base import GsshaComponentBase


class OutletHydrographComponent(GsshaComponentBase):
    """A DMI component to display the XY Series Editor showing the outlet hydrograph from the GSSHA solution."""
    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file (str): The main file associated with this component.
        """
        super().__init__(main_file)
        self._tree_commands = [
            ('Open...', '_open', ''),
        ]  # [(menu_text, menu_method, icon_file)...]

        cards = file_io_util.read_json_file(Path(main_file))
        self.sim_uuid = cards.get('SIM_UUID', '')
        self.data_file = dmi_util.find_solution_file(
            cards.get('SOLUTION_FILE_FULL', ''), cards.get('SOLUTION_FILE_RELATIVE', '')
        )

    def _open(self, query: Query, params: list[dict], win_cont: QWidget):
        """Opens the outlet hydrograph plot.

        Args:
            query: Object for communicating with GMS
            params: ActionRequest parameters
            win_cont: The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        if not self.data_file or not self.data_file.is_file():
            # message_box.message_with_ok(parent=win_cont, message='Could not find the solution file.', app_name='GMS')
            return [('ERROR', 'Could not find the solution file.')], []

        df = self._read_data_file()
        self._open_dialog(df, win_cont)
        return [], []

    def _read_data_file(self) -> pd.DataFrame:
        """Reads the data file and returns a DataFrame.

        Returns:
            pd.DataFrame: The data from the file.
        """
        df = pd.read_csv(self.data_file, sep=r'\s+', names=['Times', 'Flow'], engine='c', dtype=float)
        df.index += 1
        return df

    def _open_dialog(self, df: pd.DataFrame, win_cont: QWidget) -> None:
        """Opens the dialog with the data.

        Args:
            df: The data to display.
            win_cont: The window container.
        """
        dialog = XySeriesEditor(df, series_name='Outlet Hydrograph', parent=win_cont, read_only=True)
        dialog.exec()
