"""TextEditorComponent class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy import file_io_util
# from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.gssha.components import dmi_util
from xms.gssha.components.gssha_component_base import GsshaComponentBase
from xms.gssha.gui import gui_util


class TextEditorComponent(GsshaComponentBase):
    """A Dynamic Model Interface (DMI) component for a solution text file viewer component."""
    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file (str): The main file associated with this component.
        """
        super().__init__(main_file)
        self._tree_commands = [
            ('Open...', '_open', ''),
        ]  # [(menu_text, menu_method, icon_file)...]

        cards = file_io_util.read_json_file(Path(main_file))
        self.sim_uuid = cards.get('SIM_UUID', '')
        self.data_file = dmi_util.find_solution_file(
            cards.get('SOLUTION_FILE_FULL', ''), cards.get('SOLUTION_FILE_RELATIVE', '')
        )

    def _open(self, query, params, win_cont):
        """Opens the file in a text editor.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        if not self.data_file or not self.data_file.is_file():
            # message_box.message_with_ok(parent=win_cont, message='Could not find the solution file.', app_name='GMS')
            return [('ERROR', 'Could not find the solution file.')], []
        gui_util.open_file_in_default_app_or_notepad(self.data_file, win_cont)
        return [], []
