"""Reads the .gag file.
"""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from datetime import datetime
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.gssha.file_io import io_util


def read(file_path: Path) -> tuple[list[float], list[float], float, datetime]:
    """Reads the hyetograph stuff from PRECIP_FILE (.gag) file.

    Args:
        file_path: .gag file path.

    Returns:
        tuple[list[float], list[float], float, datetime]: x values, y values, average depth, starting date/time
    """
    x: list[float] = []
    y: list[float] = []
    nrpds: int = 0
    avg_depth: float = 1.0
    start_date_time: datetime = datetime.min
    with open(file_path, 'r') as file:
        count = 0
        for line in file:
            line = line.rstrip('\n')
            card, card_value = io_util.get_card_and_value(line)
            if card == 'ACCUM':
                date_str, value_str = card_value.rsplit(maxsplit=1)  # Exclude last word when reading the time
                date_time = io_util.datetime_from_string(date_str)

                # The start date/time comes from the hyetograph (START_DATE and START_TIME are for uniform precip)
                # so set start_date_time to the first date/time.
                if count == 0:
                    start_date_time = date_time

                minutes = io_util.relative_time_from_datetime(date_time, start_date_time)
                x.append(minutes)
                y.append(float(value_str) / avg_depth)
                count += 1
                if count >= nrpds:
                    break
            elif card == '#HYETOGRAPH':
                avg_depth = float(card_value)
                if avg_depth == 0.0:
                    avg_depth = 1.0
            elif card == 'NRPDS':
                nrpds = int(card_value)
    return x, y, avg_depth, start_date_time
