"""GsshaExporter class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import logging
import os
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util, TreeNode
from xms.guipy.dialogs import dialog_util, process_feedback_dlg, xms_parent_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread

# 4. Local modules
from xms.gssha.components import dmi_util
from xms.gssha.file_io import project_writer


def export() -> None:
    """Exports the simulation."""
    # For testing
    # from xms.guipy import debug_pause
    # debug_pause()
    # _set_test_environment()

    dialog_util.ensure_qapplication_exists()
    query = dmi_util.create_query()
    # from xms.guipy.testing import testing_tools
    # testing_tools.write_tree_to_file(query.project_tree, Path('C:/temp/project_tree.json'))
    sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, query.current_item_uuid())
    main_file = query.item_with_uuid(item_uuid=sim_node.uuid, model_name='GSSHA', unique_name='Sim_Manager').main_file
    out_dir = Path.cwd()
    parent_hwnd, main_hwnd, _icon_path = xms_parent_dlg.parse_parent_window_command_args()
    win_cont = xms_parent_dlg.get_parent_window_container(parent_hwnd)
    thread = GsshaExporter(main_file, query, out_dir, sim_node)
    process_feedback_dlg.run_feedback_dialog(thread, win_cont)


def _set_test_environment():  # pragma no cover - This is only called when manually debugging
    from xms.api.dmi import XmsEnvironment
    os.environ[XmsEnvironment.ENVIRON_RUNNING_TESTS] = 'TRUE'


class GsshaExporter(FeedbackThread):
    """Exports the simulation while showing a feedback dialog.

    The actual writing to disc is delegated to the ProjectWriter class so that this class isn't so bloated.
    """
    def __init__(self, main_file: str, query: Query, out_dir: Path, sim_node: TreeNode) -> None:
        """Initializes the class.

        Args:
            main_file (str): Simulation component main file.
            query (Query): Object for communicating with XMS
            out_dir (Path): Path to the output directory.
            sim_node (TreeNode): Simulation tree node.
        """
        super().__init__(query)
        self._main_file: str = main_file
        self._query = query
        self._out_dir = out_dir
        self._sim_node = sim_node
        self._log = logging.getLogger('xms.gssha')
        self.display_text |= {
            'title': 'Saving GSSHA Simulation',
            'working_prompt': f'Saving GSSHA simulation to \"{out_dir}\".',  # noqa: B028
            'error_prompt': 'Error(s) encountered while saving.',
            'warning_prompt': 'Warning(s) encountered while saving.',
            'success_prompt': f'Successfully exported \"{out_dir}\".',  # noqa: B028
        }

    def _run(self) -> None:
        """Exports this GSSHA simulation."""
        self._log.info('Writing simulation...')
        gssha_filename = project_writer.write(self._main_file, self._query, self._out_dir, self._sim_node)
        if gssha_filename:
            self._log = logging.getLogger('xms.gssha')
            self._log.info('Simulation export complete')
