"""Delegate for display option widgets."""

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QEvent, QSize, Qt
from PySide2.QtGui import QBrush, QFont, QPalette, QPixmap
from PySide2.QtWidgets import QDialog, QFontDialog, QPushButton, QStyle, QStyledItemDelegate

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.data.line_style import LineOptions
from xms.guipy.data.point_symbol import PointOptions
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.dialogs.line_display_options import LineDisplayOptionsDialog
from xms.guipy.dialogs.point_display_options import PointDisplayOptionsDialog
from xms.guipy.dialogs.polygon_display_options import PolygonDisplayOptionsDialog
from xms.guipy.widgets.display_option_icon_factory import DisplayOptionIconFactory


class DisplayOptionDelegate(QStyledItemDelegate):
    """Delegate for display option widgets."""
    def __init__(self, parent=None):
        """Initializes the class.

        Args:
            parent (Something derived from QObject): The parent object.
        """
        super().__init__(parent)

    def updateEditorGeometry(self, editor, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            editor (QWidget): The editor widget.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        editor.setGeometry(option.rect)

    def paint(self, painter, option, index):
        """Override of QStyledItemDelegate method of same name.

        Args:
            painter (QPainter): The painter.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if not index.isValid():
            return
        if (index.flags() & Qt.ItemIsEditable) == 0:
            dis_brush = QBrush(option.palette.window())
            painter.setBrush(dis_brush)

        if index.flags() & QStyle.State_Selected:
            sel_brush = QBrush(option.palette.highlight())
            painter.setBrush(sel_brush)

        if index.flags() & Qt.ItemIsEnabled:
            display_options = index.data(Qt.UserRole)
            btn = QPushButton()
            if isinstance(display_options, QFont):
                p = QPalette()
                font_color = index.data(Qt.DecorationRole)
                p.setColor(QPalette.ButtonText, font_color)

                btn.setPalette(p)
                btn.setFont(display_options)
                btn.setText("AaBb")
            else:
                # special button with icon
                # remove 4 from icon size so it doesn't overlap button border
                min_dim = min(option.rect.width(), option.rect.height()) - 4
                icon_size = QSize(min_dim, min_dim)
                btn.setIconSize(icon_size)
                btn.setIcon(DisplayOptionIconFactory.get_icon(display_options, min_dim))
            btn.setFixedWidth(option.rect.width())
            btn.setFixedHeight(option.rect.height())

            pix = QPixmap(option.rect.size())
            btn.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)
        else:
            pb = QPushButton()
            pb.setFixedWidth(option.rect.width())
            pb.setFixedHeight(option.rect.height())

            pix = QPixmap(option.rect.size())
            pb.render(pix)
            painter.drawPixmap(option.rect.topLeft(), pix)

    def editorEvent(self, event, model, option, index):  # noqa: N802
        """Override of QStyledItemDelegate method of same name.

        Args:
            event (QEvent): The editor event that was triggered.
            model (QAbstractItemModel): The data model.
            option (QStyleOptionViewItem): The style options.
            index (QModelIndex): The index in the model.
        """
        if index.isValid() and index.flags() & Qt.ItemIsEnabled:
            if event.type() == QEvent.MouseButtonRelease:
                display_options = index.data(Qt.UserRole)

                # check the option type, show the correct dialog
                dlg = None
                if isinstance(display_options, LineOptions):
                    dlg = LineDisplayOptionsDialog(display_options, self.parent(), True)
                elif isinstance(display_options, PointOptions):
                    dlg = PointDisplayOptionsDialog(display_options, self.parent(), True)
                elif isinstance(display_options, PolygonOptions):
                    dlg = PolygonDisplayOptionsDialog(display_options, self.parent(), True)
                elif isinstance(display_options, QFont):
                    dlg = QFontDialog(display_options, self.parent())
                if dlg and dlg.exec() == QDialog.Accepted:
                    # get values back out and set them
                    if isinstance(display_options, QFont):
                        model.setData(index, dlg.currentFont(), Qt.UserRole)
                    else:
                        model.setData(index, dlg.get_options(), Qt.UserRole)
                return True
        return super(DisplayOptionDelegate, self).editorEvent(event, model, option, index)
