"""Module for the `FeedbackThread` class and companion exceptions."""

__copyright__ = "(C) Copyright Aquaveo 2023"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
from typing import Optional, Protocol
import warnings

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query, XmsEnvironment as XmEnv

# 4. Local modules
from xms.guipy.dialogs.process_feedback_thread import ProcessFeedbackThread


class ExpectedError(Exception):
    """
    An exception that is handled as an expected error when it escapes from `FeedbackThread._run()`.

    This exception can be raised to exit from `FeedbackThread._run()` and have an error message logged at the same time.
    It should be raised like `raise ExpectedError('Something bad happened')`. The `FeedbackThread` will then handle it
    by logging the message this was constructed with ('Something bad happened', in this case) as at the `error` level.
    No stack trace will be written, since this exception is meant for expected errors, like the user specifying
    nonsensical values.
    """
    pass


class ExitError(Exception):
    """
    An exception that is handled as an already-logged error when it escapes from `FeedbackThread._run()`.

    This exception can be raised to exit from `FeedbackThread._run()`. It should be raised like `raise ExitError()`. The
    `FeedbackThread` will then suppress it. Unlike `ExpectedError`, this exception does not result in an error message
    being written. Its main use case is for when a lower level of code logs an error and reports a failure, and a higher
    level just wants to bail out as quickly as possible without logging an error twice.
    """
    pass


class Sendable(Protocol):
    """
    A protocol that FeedbackThread expects its `xms_data` parameter to support.

    Anything with a .send() method that takes no parameters is acceptable.
    """
    def send(self):
        """
        Send all the produced data to XMS.

        This should generally end up calling `Query.send()` under normal use, and probably do nothing under testing.
        """
        pass


class FeedbackThread(ProcessFeedbackThread):
    """
    A base class for background threads that run long tasks.

    To use this class, you'll want to derive something from it that sets `self.display_text` and overrides
    `self._run()`. Then you can pass instances to `xms.guipy.feedback.process_feedback_dlg.run_feedback_dialog()`.
    """
    def __init__(
        self,
        query: Optional[Query] = None,
        is_export: bool = False,
        create_query: bool = True,
        xms_data: Optional[Sendable] = None
    ):
        """
        Initialize the runner.

        Args:
            is_export: Whether this is an export script thread. If it is, export failures will be reported.
            xms_data: High-level interprocess communication object. Newer and recently refactored model interfaces
                should have an XmsData class (or use the one in xms.components), and can pass it for this parameter.
                If provided, the feedback thread will arrange for any data to be sent to XMS when appropriate. This
                saves the trouble of having to reimplement the same logic in every thread.
            query: Interprocess communication object. If `None` and `create_query` is `True`, the thread will create a
                new `Query` for derived classes automatically.

                This parameter is deprecated since it encourages user code to handle its own `Query`. Newer code should
                use `xms_data` instead.
            create_query: Whether a new `Query` should be created if `query` is `None`.

                There are feedback threads that really have no business using `Query`, such as the `ReadTemplateRunner`
                in xmshydroas, which exists only to parse a file and return some data. These runners can pass
                `create_query=False` to avoid creating a Query.

                Some tests may also find this useful to avoid using Query. If the feedback thread creates a Query under
                test, then Query will wait about 10 minutes for XMS to respond before giving up and killing the process,
                which is terrible behavior for testing. It probably isn't too hard to just put a no-op .send() method on
                the Query wrapper though, so this might not actually buy you much for tests.

                This parameter is deprecated for the same reason as `query`. Pass xms_data instead.
        """
        # This is designed for use with FeedbackDialog, which will assign our parent later, so we'll use None for now.
        super().__init__(parent=None, do_work=self._unexpected_error_handling_wrapper)

        #: Whether this is an export script.
        self.is_export: bool = is_export

        #: Messages and text to display in a feedback dialog while running the thread.
        self.display_text = {
            # Title of the dialog.
            'title': 'Running task...',
            # Display message for when the thread is running.
            'working_prompt': 'Running task, please wait...',
            # Display message for when a warning occurs.
            'warning_prompt': 'Warning(s) encountered during task. Review log output for details.',
            # Display message for when an error occurs.
            'error_prompt': 'Error(s) encountered during task. Review log output for details.',
            # Display message for when everything succeeds.
            'success_prompt': 'Task completed successfully.',
            # Text to display in a banner at the top. Banner is hidden if this is empty.
            'note': '',
            # Text to display next to the autoload checkbox. Box is hidden if this is empty.
            'auto_load': 'Close dialog automatically if successful.'
        }

        module = self.__module__
        first_two_components = module.split('.')[:2]
        self.logger_name = '.'.join(first_two_components)

        self._sender: Optional[Query | Sendable] = None
        if xms_data:
            self._sender = xms_data
        elif query:
            warnings.warn('`query` parameter is deprecated. Use `xms_data` instead.', DeprecationWarning, stacklevel=2)
            self._sender = query
        elif create_query:
            warnings.warn('`query` parameter is deprecated. Use `xms_data` instead.', DeprecationWarning, stacklevel=2)
            self._sender = Query()
        self._log: logging.Logger = logging.getLogger(self.logger_name)

    def send(self):
        """Send data to XMS, if sending is available."""
        # Scenario: The feedback thread completes successfully and must exit so the feedback dialog can report it
        # finished. As part of its work, it "helpfully" sends the data to XMS. But during its operation it reported
        # a warning. The user, upon seeing the warning, realizes something is actually horribly wrong and clicks the
        # Cancel button. But it's too late - the feedback thread already sent all the data, and the user's project will
        # be polluted with garbage no matter how many times he clicks Cancel.
        #
        # This is kept separate from `self._run()` so that the feedback dialog can decide whether to send. That way the
        # user can still cancel even after the thread finishes.
        if self._sender is not None:
            self._sender.send()

    def _expected_error_handling_wrapper(self):
        """Wrapper for self._run that lets unexpected exceptions escape the thread."""
        try:
            self._run()
        except ExpectedError as exc:
            self._log.error(str(exc))
        except ExitError:
            pass

    def _unexpected_error_handling_wrapper(self):
        """Wrapper for self._run that catches unexpected exceptions and reports a generic error instead."""
        try:
            self._expected_error_handling_wrapper()
            return
        except Exception as exc:
            # All the feedback threads I've seen swallow any exceptions before they escape and just print a generic
            # error message. It's probably to avoid scaring the user, so I replicated it here.
            self._log.error('An unexpected internal error occurred. Please contact Aquaveo tech support.')

            # The stack trace is useful for debugging though, so we'll write it somewhere for developers to find.
            XmEnv.report_error(exc)

    def test_run(self):
        """
        Run the thread under the assumption it is being run by an automated test.

        Normal running installs a last-chance exception handler that swallows all exceptions and reports a generic
        user-friendly error message. This method skips that handler and lets the exception crash the test so you get a
        developer-friendly stack trace instead.
        """
        self._expected_error_handling_wrapper()

    def _run(self):
        """Run the thread."""
        # Derived classes should override this to do any necessary work.

        # Note: Many existing feedback runners call this `_do_work()` and wrap the whole body in a `try...except...`
        # block that logs a generic error message. Derived classes should *not* imitate this. Only catch exceptions here
        # if you can do something useful with them, like try an alternative way to do the job or tell the user how to
        # fix it. The base class will take care of unexpected errors by telling the user something bad happened and
        # writing a stack trace to the Python debug log.

        # This method will have access to `self._query` if `query` or `create_query` were passed to the constructor.
        # It's recommended to pas `xms_data` to the constructor, assign it to `self._data` in your constructor, and use
        # that here instead.

        # The exceptions defined above in this file are handled by this method's caller. Details in their docstrings.

        # Testing code can call `self.test_run()` to invoke this method.
        raise ExpectedError('Thread not implemented')
