"""Dialog for viewing and editing polygon display options in XMS."""

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt, Slot
from PySide2.QtWidgets import QDialog

# 3. Aquaveo modules
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.dialogs.polygon_display_options_ui import Ui_PolygonDisplayOptionsDialog

# 4. Local modules


class PolygonDisplayOptionsDialog(QDialog):
    """A dialog to let the user set polygon display options."""
    def __init__(self, options, parent=None, change_color=True):
        """Initializes the dialog.

        Args:
            options (PolygonOptions): The color and texture to draw.
            parent (Something derived from QWidget): The parent window.
            change_color (bool): If True, the user can change the polygon color
        """
        super().__init__(parent)
        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)

        self.help_url = "https://www.xmswiki.com/wiki/SMS:Display_Options"

        self.ui = Ui_PolygonDisplayOptionsDialog()
        self.ui.setupUi(self)
        self.ui.textures.set_texture(options.texture)
        self.ui.textures.draw_all_textures()
        if change_color:
            self.ui.color_btn.set_color(options.color)
        else:  # Hide color selector widgeta if disabled.
            self.ui.color_btn.hide()
            self.ui.lab_line_color.hide()
        self.setMinimumSize(self.size())

    def get_options(self):
        """Returns the current display options.

        Returns:
            options (PolygonOptions): The current display options.
        """
        options = PolygonOptions()
        options.color = self.get_color()
        options.texture = self.get_texture()
        return options

    def get_color(self):
        """Returns the current color.

        Returns:
            color (QColor): The current color.
        """
        return self.ui.color_btn.color

    def get_texture(self):
        """Returns the current symbol.

        Returns:
            texture (PolygonTexture): The current texture.
        """
        return self.ui.textures.get_texture()

    @Slot()
    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    @Slot()
    def accept(self):
        """Called when the OK button is clicked."""
        QDialog.accept(self)

    @Slot()
    def reject(self):
        """Called when the Cancel button is clicked."""
        QDialog.reject(self)
