"""A dataset tree item selector dialog."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QAbstractItemView, QListWidgetItem

# 3. Aquaveo modules
import xms.api._xmsapi.dmi as xmd
from xms.api.tree import tree_util

# 4. Local modules
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.treeitem_selector_datasets_ui import Ui_dlg_treeitem_selector_datasets
from xms.guipy.time_format import XmsTimeFormatter


class TreeItemSelectorDatasetsDlg(TreeItemSelectorDlg):
    """A dialog that appears when selecting a tree item customized for datasets."""
    def __init__(
        self,
        title,
        pe_tree,
        selected_dataset='',
        selected_time_step=0,
        query=None,
        allow_all_time_steps=False,
        override_icon=None,
        show_root=False,
        parent=None,
        allow_change_selection=True
    ):
        """Initializes the class, sets up the ui.

        Args:
            title (str): The dialog title
            pe_tree (TreeNode): Root of the project explorer tree to display
            selected_dataset (str): UUID of the dataset that will be selected when the dialog appears.
            selected_time_step (int): Index of the selected time step (0-based).
            query (xms.api.dmi.Query): Object for communicating with GMS.
            allow_all_time_steps (bool): If true, an 'All time steps' toggle is displayed.
            override_icon (callable): Callable method to provide icons that override the default for a tree item
                type. Method should take a TreeNode and return an icon path if it is overriden, else empty string. Icon
                path must be in whatever format that works in the calling package.
            show_root (bool): If True, root of the tree will be shown
            parent (Something derived from QWidget): The parent window.
            allow_change_selection (bool): Whether to allow changing the selected dataset. If this is True, only
                the time step can be selected for the selected_dataset.
        """
        super().__init__(
            title,
            xmd.DatasetItem,
            pe_tree,
            selected_dataset,
            override_icon,
            show_root,
            parent,
            ui_class=Ui_dlg_treeitem_selector_datasets(),
            allow_change_selection=allow_change_selection
        )
        self._time_settings = None
        self._pe_tree = pe_tree
        self.selected_time_step = selected_time_step
        self.allow_all_time_steps = allow_all_time_steps
        self._allow_change_selection = allow_change_selection
        self._dset_dumps = {}  # Cache of datasets that have been clicked on

        # Set up the query
        self._query = query

        # Setup the ui
        self.setWindowTitle(title)
        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)

        # Signals
        self.ui.tree_pe_selector.itemChanged.connect(self._on_item_changed)
        self.ui.tog_all_time_steps.stateChanged.connect(self._on_tog_all_time_steps)

        if allow_all_time_steps:
            self.ui.tog_all_time_steps.setChecked(True)
        else:
            self.ui.tog_all_time_steps.hide()
            self.ui.lst_time_steps.setEnabled(True)

        # Time steps
        self.ui.lst_time_steps.setSelectionMode(QAbstractItemView.SingleSelection)
        self.setup_time_step_list(True)
        if self.ui.lst_time_steps.count() > 0:
            self.ui.lst_time_steps.item(self.selected_time_step).setSelected(True)
            self.ui.lst_time_steps.setCurrentRow(self.selected_time_step)

    def _on_item_changed(self, item, column):
        """Slot called when itemChanged signal sent from tree control.

        Args:
            item (QTreeWidgetItem): The item that was changed.
            column (int): The column.
        """
        self.setup_time_step_list()

    def _on_tog_all_time_steps(self, state):
        """Slot called when stateChanged signal sent from toggle box.

        Args:
            state (int): The new state.
        """
        self.ui.lst_time_steps.setEnabled(state == 0)

    def setup_time_step_list(self, first=False):
        """Sets up the list of time steps."""
        if not self._allow_change_selection and not first:
            # Don't reset the time step window if it's already set up and you can't change the selected dataset
            return
        self.ui.lst_time_steps.clear()
        uuid = self.ui.tree_pe_selector.get_selected_item_uuid()
        node = tree_util.find_tree_node_by_uuid(self._pe_tree, uuid)
        if not node:  # Hide the time step window if no dataset is selected
            self.ui.grp_time_steps.hide()
            return

        times = self._get_dataset_times(node)
        if not times:  # Hide the time step window if dataset is steady-state
            self.ui.grp_time_steps.hide()
            return

        self.ui.grp_time_steps.show()
        for time in times:
            new_item = QListWidgetItem(str(time), self.ui.lst_time_steps)
            self.ui.lst_time_steps.addItem(new_item)

        self.ui.lst_time_steps.item(0).setSelected(True)
        self.ui.lst_time_steps.setCurrentRow(0)

    def _get_dataset_times(self, node):
        """Gets the list of time step times from the dataset.

        Args:
            node (TreeNode): The dataset

        Returns:
            list of time step times.
        """
        if node.num_times < 2:
            return []  # Steady-state dataset

        self._query_for_time_settings()  # Get the XMS global time settings if we haven't already
        dataset = self._get_dataset(node.uuid)
        times = self._times_from_dataset(dataset)
        return times

    def _get_dataset(self, dset_uuid):
        """If we don't already have this dataset, uses the query to find it and cache it.

        Args:
            dset_uuid (str): The uuid of the selected item.

        Returns:
            The dataset with the given uuid.
        """
        if dset_uuid not in self._dset_dumps:
            self._dset_dumps[dset_uuid] = self._query.item_with_uuid(dset_uuid)
        return self._dset_dumps[dset_uuid]

    def _query_for_time_settings(self):
        """Uses the query to get the current XMS global time settings."""
        if self._time_settings is None:
            self._time_settings = XmsTimeFormatter(self._query.global_time_settings)

    def _times_from_dataset(self, dataset):
        """Gets the list of time steps times from the dataset.

        Args:
            dataset (DatasetReader): The data_objects dataset dump

        Returns:
            list of time step times.
        """
        if not dataset:
            return []

        ts_times_strings = []
        # Set reference time if there is one in case there is no zero time
        self._time_settings.ref_time = dataset.ref_time if dataset.ref_time is not None else None

        for ts_idx in range(dataset.num_times):
            ts_delta = dataset.timestep_offset(ts_idx)
            ts_times_strings.append(self._time_settings.format_time(ts_delta))

        return ts_times_strings

    def get_selected_time_step_string(self):
        """Returns the selected time step string prefixed with a ':'.

        1-based.

        Returns:
            (str):
                ':all' if the 'All time steps' toggle is visible and checked

                ':1' if there is no dataset selected

                ':n' if there is a dataset and time step selected where n is the 1-based time step index

                '' if there is a steady-state dataset selected
        """
        if self.allow_all_time_steps and self.ui.tog_all_time_steps.isChecked():
            return ':all'
        elif self.ui.lst_time_steps.currentRow() >= 0:
            return f':{self.ui.lst_time_steps.currentRow() + 1}'
        elif not self.ui.grp_time_steps.isVisible():  # Steady-state dataset, only one time step
            return ''
        else:
            return ':1'

    def get_selected_time_step_index(self):
        """Returns the selected time step index for the selected dataset or -1 if any problem.

        0-based.

        Returns:
            (int):
                -1 if the 'All time steps' toggle is visible and checked.

                -1 if there is no dataset selected

                n if there is a dataset and time step selected where n is the 0-based time step index
        """
        if self.allow_all_time_steps and self.ui.tog_all_time_steps.isChecked():
            return -1
        elif self.ui.lst_time_steps.currentRow() >= 0:
            return self.ui.lst_time_steps.currentRow()
        elif not self.ui.grp_time_steps.isVisible():  # Steady-state dataset, only one time step
            return 0
        else:
            return -1

    def get_selected_item_uuid(self):
        """Returns the selected item's uuid or None if nothing selected.

        Returns:
            See description.
        """
        return self.ui.tree_pe_selector.get_selected_item_uuid()


def uuid_and_time_step_index_from_string(uuid_and_ts):
    """Given a uuid string that may have a time step number appended after a ':', returns the uuid and ts index.

    Used with datasets. The number is assumed to be 1-based, but the returned index will be 0-based.

    Args:
        uuid_and_ts (str): A string like 'd0dc9cb1-ab7e-4fff-bbf5-6eeda674a255' or
            'd0dc9cb1-ab7e-4fff-bbf5-6eeda674a255:12'

    Returns:
        (tuple): tuple containing:

            uuid (str): The dataset uuid.

            ts_idx (int): 0-based timestep index.
    """
    # Get uuid and time step index
    words = uuid_and_ts.split(':')
    uuid = ''
    ts_idx = 0
    try:
        uuid = words[0]
        ts_idx = int(words[1]) - 1  # May or may not have an integer time step index at end of string
    except Exception:
        pass
    return uuid, ts_idx
