"""Notes class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime
import os
import sqlite3

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.time_format import ISO_DATETIME_FORMAT


class Notes:
    """Handles reading and writing notes."""
    def __init__(self, db_filename='', xms=''):
        """Initializes the class.

        Args:
            db_filename (str): Path to the notes database file. ONLY USED WHEN TESTING.
            xms (str): Name of the XMS app ('GMS', 'SMS', 'WMS). ONLY USED WHEN TESTING.
        """
        self._db_filename = os.environ.get('XMS_PYTHON_APP_NOTES_DATABASE') if not db_filename else db_filename
        self._xms = os.environ.get('XMS_PYTHON_APP_NAME', 'XMS') if not xms else xms

    def add_note(self, uuid, note):
        """Adds a note.

        Args:
            uuid (str): UUID of an object.
            note (str): The note.
        """
        if not self._db_filename:
            raise RuntimeError('Notes database not found.')

        conn = sqlite3.connect(self._db_filename)
        cur = conn.cursor()
        date_time = datetime.now().strftime(ISO_DATETIME_FORMAT)
        cur.execute(
            "INSERT INTO Notes(Selected, uuid, Type, DateTime, Note) VALUES(?, ?, ?, ?, ?);",
            (0, uuid, self._xms, date_time, note)
        )
        conn.commit()
        conn.close()

    def get_notes(self, uuid):
        """Returns all the notes associated with the given uuid.

        Args:
            uuid (str): UUID of an object.

        Returns:
            list(list[tuple(str,str)]): List of tuples containing the date string and note string
            for each row with specified UUID.
        """
        if not self._db_filename:
            raise RuntimeError('Notes database not found.')

        conn = sqlite3.connect(self._db_filename)
        cur = conn.cursor()
        cur.execute("SELECT Type, DateTime, Note FROM Notes WHERE uuid=?;", (uuid, ))
        results = cur.fetchall()
        conn.commit()
        conn.close()
        return results
