"""The Assign EWN Feature polygon attribute dialog."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtWidgets import QDialogButtonBox, QHBoxLayout, QScrollArea, QVBoxLayout, QWidget

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.param.param_qt_helper import ParamQtHelper

# 4. Local modules


class ParamDialogBase(XmsDlg):
    """A base class used for creating simple dialogs with Param tech."""
    def __init__(self, parent, param_data_class, param_layout, window_title, help_url, dialog_name):
        """Initializes the dialog, sets up the ui.

        Args:
            parent (QWidget): Parent window.
            param_data_class (derived from param.Parameterized): A class derived from param.Parameterized.
            param_layout: A class holding a dict with ParamLayout objects. Used to extend the basic param functionality.
            window_title (str): The window title.
            help_url (str): URL to the help web page tied to the Help button.
            dialog_name (str): Something like 'xmssrh.gui.summary_report_dialog' used by XmsDlg for the registry.
        """
        super().__init__(parent, dialog_name)
        self._help_url = help_url
        self._param_data_class = param_data_class
        self._widgets = dict()
        self._param_helper = ParamQtHelper(self)
        self.setup_ui(param_layout, window_title)

    def setup_ui(self, param_layout, window_title):
        """Setup the dialog and its widgets.

        Args:
            param_layout: A class holding a dict with ParamLayout objects.
            window_title (str): The window title.
        """
        self._param_data_class.param_layout = param_layout
        self.setWindowTitle(window_title)
        self._set_layout('', 'top_layout', QVBoxLayout())
        self._widgets['scroll_area'] = QScrollArea(self)
        self._widgets['top_layout'].addWidget(self._widgets['scroll_area'])
        self._widgets['scroll_area'].setWidgetResizable(True)
        self._widgets['scroll_area'].setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOn)
        # needed for the scroll area to function properly (see stackoverflow)
        self._widgets['scroll_area_widget'] = QWidget()
        self._set_layout('scroll_area_widget', 'scroll_area_layout', QVBoxLayout())
        self._widgets['scroll_area'].setWidget(self._widgets['scroll_area_widget'])
        self._param_helper.add_params_to_layout(self._widgets['scroll_area_layout'], self._param_data_class)
        self._widgets['scroll_area_layout'].addStretch()

        self._widgets['btn_box'] = QDialogButtonBox()
        self._widgets['top_layout'].addWidget(self._widgets['btn_box'])

        # set all widget values and hide/show
        self._param_helper.do_param_widgets(None)

        # QDialogButtonBox with Ok and Cancel buttons
        self._widgets['btn_box'].setOrientation(Qt.Horizontal)
        self._widgets['btn_box'].setStandardButtons(
            QDialogButtonBox.Cancel | QDialogButtonBox.Ok | QDialogButtonBox.Help
        )
        self._widgets['btn_box'].accepted.connect(self.accept)
        self._widgets['btn_box'].rejected.connect(self.reject)
        self._widgets['btn_box'].helpRequested.connect(self.help_requested)

    def _set_layout(self, parent_name, layout_name, layout):
        """Adds a layout to the parent.

        Args:
            parent_name (str): Name of parent widget in self.widgets or '' for self
            layout_name (QLay): Name of layout in parent widget
            layout (str): QtLayout to be used
        """
        self._widgets[layout_name] = layout
        if parent_name:
            parent = self._widgets[parent_name]
        else:
            parent = self
        if type(parent) in [QVBoxLayout, QHBoxLayout]:
            parent.addLayout(self._widgets[layout_name])  # pragma nocover
        else:
            parent.setLayout(self._widgets[layout_name])

    def help_requested(self):  # pragma nocover
        """Called when the Help button is clicked."""
        webbrowser.open(self._help_url)
