"""Imlementation of QtCharts.QChartView."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCharts import QtCharts
from PySide2.QtCore import QRectF, Qt
from PySide2.QtGui import QCursor

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.widgets import widget_builder


class QxChartView(QtCharts.QChartView):
    """Custom QxChartView for the XySeriesEditor."""
    def __init__(self, chart):
        """Initializes the class.

        Args:
            chart (QChart): The chart.
        """
        super().__init__(chart)

        self.setRubberBand(QtCharts.QChartView.RectangleRubberBand)  # Enables zoom to rectangle
        self.setContextMenuPolicy(Qt.CustomContextMenu)
        self.customContextMenuRequested.connect(self.on_right_click)

    def on_right_click(self, point):
        """Called when the right mouse is clicked and displays a context menu.

        Args:
            point (QPoint): Location of the mouse clidk

        """
        menu_list = [['icons/frame.svg', 'Frame', self.on_frame]]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.mapToGlobal(point))

    def on_frame(self):
        """Frames the plot in the display."""
        self.chart().zoomReset()

    def wheelEvent(self, event):  # noqa: N802
        """Zooms on mouse wheel.

        See https://stackoverflow.com/questions/48623595

        Args:
            event (Qt.QEvent): The mouse wheel event to handle
        """
        if event.delta() > 0:
            factor = 2.0
        else:
            factor = 0.5

        r = QRectF(
            self.chart().plotArea().left(),
            self.chart().plotArea().top(),
            self.chart().plotArea().width() / factor,
            self.chart().plotArea().height() / factor
        )
        mouse_pos = self.mapFromGlobal(QCursor.pos())
        r.moveCenter(mouse_pos)
        self.chart().zoomIn(r)
        delta = self.chart().plotArea().center() - mouse_pos
        self.chart().scroll(delta.x(), -delta.y())

    def mouseReleaseEvent(self, event):  # noqa: N802
        """Overridden to prevent right-click from zooming out.

        See https://stackoverflow.com/questions/40783089

        Args:
            event (Qt.QEvent): The mouse wheel event to handle
        """
        if event.button() == Qt.RightButton:
            return  # event doesn't go further
        super().mouseReleaseEvent(event)  # any other event
