"""QCombobox with editable options."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtCore import QModelIndex, Qt
from PySide2.QtGui import QStandardItemModel
from PySide2.QtWidgets import QComboBox

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.validators.cbx_number_corrector import CbxNumberCorrector
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

NO_QMODELINDEX = QModelIndex()


class QxEditableComboBoxModel(QStandardItemModel):
    """Model for combobox where the last item is editable."""
    def __init__(self, strings, edit_indices, parent=None):
        """Initializes the class.

        Args:
            strings (list): List of the combobox option strings
            edit_indices (set): Set of the combobox item indices that are editable
            parent (object): Qt parent
        """
        super().__init__(parent)
        self.strings = strings
        self._editable_indices = edit_indices

    def index(self, row, column, parent=NO_QMODELINDEX):
        """Get a Qt model index for a specified row and column.

        Args:
            row (int): Row of item
            column (int): Column of item
            parent (QObject): The Qt parent

        Returns:
            (QModelIndex): The Qt model index
        """
        return self.createIndex(row, column)

    def rowCount(self, index=NO_QMODELINDEX):  # noqa: N802
        """Get the number of rows in the model.

        Args:
            index (QModelIndex): Item to get row count of, self by default.

        Returns:
            (int): See description
        """
        return len(self.strings)

    def columnCount(self, index=NO_QMODELINDEX):  # noqa: N802
        """Get the number of columns in the model.

        Args:
            index (QModelIndex): Item to get column count of, self by default.

        Returns:
            (int): See description
        """
        return 1

    def data(self, index, role=Qt.DisplayRole):
        """Get data stored in the model for a specified item and role.

        Args:
            index (QModelIndex): Item to get data for
            role (Qt.ItemDataRole): The role to get data for

        Returns:
            (object): The data for the specified item and role
        """
        if role == Qt.DisplayRole or role == Qt.EditRole:
            if index.row() < len(self.strings):
                return self.strings[index.row()]
        return None

    def setData(self, index, value, role=Qt.EditRole):  # noqa: N802
        """Set data stored in the model for a specified item and role.

        Args:
            index (QModelIndex): Item to set data for
            value (object): The data for the specified item and role
            role (Qt.ItemDataRole): The role to set data for

        Returns:
            (bool): True, if model data updated
        """
        if role == Qt.EditRole:
            if index.row() in self._editable_indices:  # Item is editable
                if self.strings[index.row()] != value:
                    self.strings[index.row()] = value
                    self.dataChanged.emit(index, index)
                return True
        return False

    def flags(self, index):
        """Get the Qt model flags for a specified item.

        Args:
            index (QModelIndex): The item to get flags of

        Returns:
            (Qt.ItemFlag): Flags set on the specified item

        """
        f = Qt.ItemIsSelectable | Qt.ItemIsEnabled
        if index.row() in self._editable_indices:  # Item is editable
            f = f | Qt.ItemIsEditable
        return f


class QxEditableComboBox(QComboBox):
    """Combobox where the last item is editable."""
    def __init__(self, parent=None, strings=None, edit_indices=None):
        """Initializes the class.

        Args:
            strings (list): List of the combobox option strings
            edit_indices (set): Set of the combobox item indices that are editable
            parent (object): Qt parent

        """
        super().__init__(parent)
        # Setup the combobox model
        editable_indices = edit_indices if edit_indices else set()
        items = strings if strings else []
        self.setModel(QxEditableComboBoxModel(parent=self, strings=items, edit_indices=editable_indices))

        self.currentIndexChanged.connect(self.on_current_index_changed)
        self.setEditable(True)
        self.num_validator = QxDoubleValidator(parent=self)
        self.num_validator.setDecimals(CbxNumberCorrector.DEFAULT_PRECISION)
        self.corrector = CbxNumberCorrector(self)
        self.corrector.set_editable_items(editable_indices)
        self.setValidator(self.num_validator)
        # self.editTextChanged.connect(self.on_edit_text_changed)
        self.lineEdit().editingFinished.connect(self.on_editing_finished)
        self.installEventFilter(self.corrector)

    def on_editing_finished(self):
        """Called when user is done editing text."""
        self.on_edit_text_changed(self.lineEdit().text())

    def on_edit_text_changed(self, text):
        """Update text in combobox model when editable option is changed."""
        if self.model():
            # set data to model immediately
            index = self.model().index(self.currentIndex(), 0)
            self.model().setData(index, text)

    def on_current_index_changed(self, index):
        """Enable/disable editing of the combobox when option is changed."""
        if self.model():
            # disable editing if model disable it
            flags = self.model().flags(self.model().index(index, 0))
            if flags & Qt.ItemIsEditable and self.lineEdit():
                self.corrector.default_value = self.lineEdit().text()
                self.lineEdit().setReadOnly(False)
            elif self.lineEdit():
                self.lineEdit().setReadOnly(True)
