"""Module for the QxFilePickerWidget."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtCore import QEvent, QObject
from PySide2.QtGui import QFontMetrics
from PySide2.QtWidgets import QLineEdit, QWidget

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.dialogs.file_selector_dialogs import get_open_filename, get_open_foldername, get_save_filename
from xms.guipy.widgets import widget_builder
from xms.guipy.widgets.qx_file_picker_widget_ui import Ui_QxFilePicker


class EventFilter(QObject):
    """Event filter object used to detect a resize event."""
    def __init__(self, widget=None, *args):
        """Initializes the class."""
        QObject.__init__(self, *args)
        self.widget = widget

    def eventFilter(self, obj, event) -> bool:  # noqa N802 (function name 'eventFilter' should be lowercase)
        """Gets called for every event this object is connected to.

        Args:
            obj: Qt object associated with the event.
            event: The QEvent object.

        Returns:
            (bool): True if the event was handled.
        """
        if event.type() == QEvent.Resize:
            self.widget.on_resize()
        return QObject.eventFilter(self, obj, event)


class QxFilePickerWidget(QWidget):
    """Widget for picking a file."""
    def __init__(
        self,
        parent: QWidget,
        initial_path: str = '',
        file_filter: str = '*.*',
        file_filters: str = '*.*',
        pick_folder: bool = False,
        is_output: bool = False,
        read_only: bool = False
    ):
        """
        Initialize the widget.

        Args:
            parent: Parent widget.
            initial_path: The path to set initially.
            file_filter: Initially selected filter to use when picking files.
            file_filters: List of all available filters to use when picking files.
            pick_folder: Whether this widget should pick a folder, rather than a file.
            is_output: Whether the widget should warn the user if the file will be overwritten, as opposed to that the
                file does not exist.
            read_only: If True, edit field is made read only and displayed path may be shortened to fit.
        """
        super().__init__(parent)

        self.pick_folder = pick_folder
        self.is_output = is_output
        self.file_filter = file_filter
        self.file_filters = file_filters
        self._read_only = read_only
        self._event_filter = None

        self.ui = Ui_QxFilePicker()
        self.ui.setupUi(self)
        if read_only:
            # Make edit field read only and install event filter to catch resize events
            widget_builder.make_lineedit_readonly(self.ui.file_path)
            self._event_filter = EventFilter(widget=self)
            self.ui.file_path.installEventFilter(self._event_filter)

        self._selected_path = ''  # Used only when read_only is True
        self.selected_path = initial_path

        self.ui.browse_button.clicked.connect(self.browse)

    @property
    def selected_path(self) -> str:
        """The currently selected file path, or an empty string if none."""
        if self._read_only:
            return self._selected_path
        else:
            return self.ui.file_path.text()

    @selected_path.setter
    def selected_path(self, path: str):
        """The currently selected file path, or an empty string if none."""
        if self._read_only:
            self._selected_path = path
            self.ui.file_path.setText(adjust_path_to_fit(path, self.ui.file_path))
        else:
            self.ui.file_path.setText(path)

    def browse(self):
        """Slot for the Browse button."""
        initial_path = self.ui.file_path.text()
        if self.pick_folder:
            new_path = get_open_foldername(self, self.caption, initial_path)
        elif self.is_output:
            new_path = get_save_filename(self, self.file_filter, self.file_filters, start_dir=initial_path)
        else:
            new_path = get_open_filename(self, 'Select file', self.file_filter, initial_path)
        self.selected_path = str(Path(new_path).resolve())

    def on_resize(self):
        """If self._event_filter is defined, called when widget is resized."""
        if self._read_only:
            self.selected_path = self._selected_path  # Trigger code that may shorten the path


def _text_width(text: str, font_metrics: QFontMetrics) -> int:
    """Return the text width given the font metrics.

    Args:
        text: The text.
        font_metrics: The font metrics.

    Returns:
        See description.
    """
    return font_metrics.horizontalAdvance(text)


def _available_width(lineedit: QLineEdit) -> int:
    """Return the available width for text in the QLineEdit.

    Args:
        lineedit: The QLineEdit.

    Returns:
        See description.
    """
    return lineedit.width() - (lineedit.contentsMargins().left() + lineedit.contentsMargins().right())


def adjust_path_to_fit(text: str, lineedit: QLineEdit) -> str:
    """Return shortened path by replacing path parts with '...' until it fits in the QLineEdit or we run out parts.

    The returned path may still be too long to fit in the QLineEdit.

    Args:
        text: The path.
        lineedit: The QLineEdit.

    Returns:
        See description.
    """
    font_metrics = lineedit.fontMetrics()
    path = Path(text)
    parts = path.parts
    last_part = path.name
    replacement_text = '...'
    part_idx = len(parts) - 1
    text_width = _text_width(text, font_metrics)
    available_width = _available_width(lineedit)
    new_text = text
    while text_width > available_width and part_idx > 1:
        part_idx -= 1
        parts_text = str((Path().joinpath(*parts[:part_idx])).resolve())
        new_text = f'{parts_text}{replacement_text}{last_part}'
        text_width = _text_width(new_text, font_metrics)
    return new_text
