"""Exports an unstructured grid to a format HEC-RAS can read."""
# 1. Standard python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.constraint import read_grid_from_file

# 4. Local modules
from xms.hecras._MeshGenerator import MeshGenerator


class HecrasGeomWriter:
    """Writes HEC-RAS .g01 ASCII geometry, .g01.hdf geometry, .prj project and .prj projection files."""

    def __init__(self, xms_data=None):
        """Initializes HecrasGeomWriter writer."""
        self._project_name = 'project_name'
        self._do_ugrid = None
        self._ugrid = None
        if xms_data:  # testing
            self._project_name = xms_data['project_name']
            self._do_ugrid = xms_data['do_ugrid']
        else:  # pragma: no cover
            self._get_xms_data()

    def _get_xms_data(self):  # pragma: no cover
        """Retrieve the simulation's geometry as a data_object and the project name, if there is one.

        Returns:
            tuple(xms.data_objects.parameters.UGrid, str): The simulation geometry and the project name.
                'project_name' if the SMS project has not been saved.
        """
        query = Query()
        if query.xms_project_path:  # Way the interface is currently set up, the project must be saved before export.
            self._project_name = os.path.splitext(os.path.basename(query.xms_project_path))[0]
        sim_uuid = query.current_item_uuid()
        sim_item = tree_util.find_tree_node_by_uuid(query.project_tree, sim_uuid)
        ugrid_items = tree_util.descendants_of_type(sim_item, xms_types=['TI_UGRID_PTR'], allow_pointers=True,
                                                    recurse=False)
        if ugrid_items:
            self._do_ugrid = query.item_with_uuid(ugrid_items[0].uuid)

    def _get_cell_definition(self, cell_idx):
        """Get a mesh cell's definition in the format required by Herman's code.

        Args:
            cell_idx (int): 0-based cell index to get definition for

        Returns:
            list: [cell_id, [pt1_id, ..., ptN_id]] - Note that the cell and point ids are 1-based in return list.
        """
        return [cell_idx + 1, [pt_idx + 1 for pt_idx in self._ugrid.get_cell_points(cell_idx)]]

    def _write_projection_file(self):
        """Writes the mesh's well known text to a .prj projection file."""
        with open(f'{self._do_ugrid.name}_projection.prj', 'w') as f:
            f.write(self._do_ugrid.projection.well_known_text)

    def write(self):
        """Top-level function to trigger the read of HEC-RAS input files."""
        # get the mesh's projection units.
        horiz_units = "FOOT" if self._do_ugrid.projection.horizontal_units.startswith('FEET') else 'SI Units'

        # Read the mesh from the CoGrid file
        co_grid = read_grid_from_file(self._do_ugrid.cogrid_file)
        self._ugrid = co_grid.ugrid

        # get polys
        polys = [self._get_cell_definition(cell_idx) for cell_idx in range(self._ugrid.cell_count)]

        # get points
        locs = self._ugrid.locations
        points = [[i + 1, [locs[i][0], locs[i][1]]] for i in range(len(locs))]

        work_dir = os.getcwd() + "\\"
        work_dir = work_dir.replace("\\", "/")
        mesh_gen = MeshGenerator(polys, points, horiz_units, work_dir, self._project_name,
                                 flowareaname=self._do_ugrid.name)
        mesh_gen.process()
        self._write_projection_file()
