"""Defines the boundary conditions."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Curve, GenericModel, Group, Section
from xms.tool_core.table_definition import FloatColumnType, InputFileColumnType, TableDefinition

# 4. Local modules
from xms.hgs.data.domains import Domains

max_min_sheets = 'Max/min node sheets'
top_bottom_elevs = 'Top/bottom elevations'
range_opts = [max_min_sheets, top_bottom_elevs]  # Vertical range options


class InputType:
    """The input types."""
    CONSTANT = 'Constant'
    TIME_SERIES = 'Time series'
    TIME_RASTERS = 'Time rasters'


def create() -> GenericModel:
    """Creates and returns the generic model.

    Returns:
        (GenericModel): The generic model.
    """
    creator = BcGenericModel()
    return creator.create()


class BcGenericModel:
    """Creates the GenericModel for the BC coverage.

    The only reason I wanted to use a class for this is so that I could easily reuse the descriptions via a member
    variable instead of a function call or global variable or a parameter that keeps getting passed through multiple
    functions.
    """
    def __init__(self):
        """Initializes the class."""
        self._desc = _descriptions()

    def create(self) -> GenericModel:
        """Creates and returns the generic model.

        Returns:
            (GenericModel): The generic model.
        """
        gm = GenericModel(
            exclusive_point_conditions=True, exclusive_arc_conditions=True, exclusive_polygon_conditions=True
        )
        self._add_point_bcs(gm)
        self._add_arc_bcs(gm)
        self._add_polygon_bcs(gm)
        return gm

    def _create_and_init(self, section: str, gmi_section: Section, group_name: str, domains: list[str] | None) -> Group:
        """Creates the group and initializes it with the things that are standard: name etc.

        Args:
            section (str): 'point_parameters', 'arc_parameters' etc.
            gmi_section (Section): The group set added to.
            group_name (str): Name of the group (BC).
            domains (List[str] | None): List of allowed domains.

        Returns:
            (Group): The group.
        """
        desc = self._desc
        gp = gmi_section.add_group(group_name=group_name, label=group_name, description=desc[group_name])

        gp.add_text('name', 'Name', default=_defaultize(group_name), description=desc['name'], required=True)

        if domains and len(domains) > 1:
            gp.add_option('domain', 'Domain', default=domains[0], options=domains, description=desc['domain'])

        if domains and Domains.PM in domains:
            parent_domain = [section, group_name, 'domain']
            range_opts = ['Max/min node sheets', 'Top/bottom elevations']
            p = gp.add_option(
                'range_opts',
                'Vertical range',
                default=range_opts[0],
                options=range_opts,
                description=desc['range_opts']
            )
            p.add_dependency(parent=parent_domain, flags={Domains.PM: True, Domains.OLF: False})
            parent_range = [section, group_name, 'range_opts']

            p = gp.add_integer('max_sheet', 'Maximum sheet', default=1, description=desc['max_sheet'])
            p.add_dependency(parent=parent_range, flags={range_opts[0]: True, range_opts[1]: False})
            p = gp.add_integer('min_sheet', 'Minimum sheet', default=1, description=desc['min_sheet'])
            p.add_dependency(parent=parent_range, flags={range_opts[0]: True, range_opts[1]: False})

            p = gp.add_float('top_elev', 'Top elevation', default=0.0, description=desc['top_elev'])
            p.add_dependency(parent=parent_range, flags={range_opts[0]: False, range_opts[1]: True})
            p = gp.add_float('bottom_elev', 'Bottom elevation', default=0.0, description=desc['bottom_elev'])
            p.add_dependency(parent=parent_range, flags={range_opts[0]: False, range_opts[1]: True})
        return gp

    def _add_simple_bc(self, section: str, gmi_section: Section, group_name: str, domains: list[str] | None) -> Group:
        """Adds a simple bc like 'head', 'simple river'.

        Args:
            section (str): 'point_parameters', 'arc_parameters' etc.
            gmi_section (Section): The group set added to.
            group_name (str): Name of the group (BC).
            domains (List[str] | None): List of allowed domains.

        Returns:
            (Group): The group.
        """
        gp = self._create_and_init(section, gmi_section, group_name, domains)
        desc = self._desc

        # Add input type stuff
        options = ['Constant', 'Time series', 'Time rasters']
        gp.add_option('input_type', 'Input type', default=options[0], options=options, description=desc['input_type'])
        parent = [section, group_name, 'input_type']

        p = gp.add_float('constant', 'Constant', default=0.0, description=desc['constant'])
        p.add_dependency(parent=parent, flags={'Constant': True, 'Time series': False, 'Time rasters': False})

        p = gp.add_checkbox_curve(
            'xy_series',
            'Time series',
            checkbox_label='Interpolate',
            axis_titles=['Time', 'Flux'],
            modes=(Curve.STAIRS, Curve.CURVE),
            default_check_state=False,
            description=desc['xy_series'],
            checkbox_description=desc['interpolate']
        )
        p.add_dependency(parent=parent, flags={'Constant': False, 'Time series': True, 'Time rasters': False})

        table_def = TableDefinition(
            [
                FloatColumnType(header='Time', default=0.0, tool_tip='Time'),
                InputFileColumnType(header='Raster', default='', file_filter='*.*', tool_tip='Raster file')
            ]
        )
        p = gp.add_table(
            'time_raster_table',
            'Time raster table',
            default=[],
            table_definition=table_def,
            description='Time raster table'
        )
        p.add_dependency(parent=parent, flags={'Constant': False, 'Time series': False, 'Time rasters': True})

        return gp

    def _add_arc_bc(self, gmi_section: Section, group_name: str, domains: list[str]) -> None:
        """Adds an arc bc with upstream and downstream inputs.

        Args:
            gmi_section (Section): The group set added to.
            group_name (str): Name of the group (BC).
            domains (List[str] | None): List of allowed domains.
        """
        gp = self._create_and_init('arc_parameters', gmi_section, group_name, domains)
        desc = self._desc

        # Add input type stuff
        options = ['Constant', 'Time series']
        gp.add_option('input_type', 'Input type', default=options[0], options=options, description=desc['input_type'])
        parent = ['arc_parameters', group_name, 'input_type']

        p = gp.add_float('constant_start', 'Constant (start)', default=0.0, description=desc['constant_start'])
        p.add_dependency(parent=parent, flags={'Constant': True, 'Time series': False})

        p = gp.add_checkbox_curve(
            'xy_series_start',
            'Time series (start)',
            checkbox_label='Interpolate',
            axis_titles=['Time', 'Flux'],
            modes=(Curve.STAIRS, Curve.CURVE),
            default_check_state=False,
            description=desc['xy_series_start'],
            checkbox_description=desc['interpolate']
        )
        p.add_dependency(parent=parent, flags={'Constant': False, 'Time series': True})

        p = gp.add_float('constant_end', 'Constant (end)', default=0.0, description=desc['constant_end'])
        p.add_dependency(parent=parent, flags={'Constant': True, 'Time series': False})

        p = gp.add_checkbox_curve(
            'xy_series_end',
            'Time series (end)',
            checkbox_label='Interpolate',
            axis_titles=['Time', 'Flux'],
            modes=(Curve.STAIRS, Curve.CURVE),
            default_check_state=False,
            description=desc['xy_series_end'],
            checkbox_description=desc['interpolate']
        )
        p.add_dependency(parent=parent, flags={'Constant': False, 'Time series': True})

        gp.add_boolean('map_to_boundary', 'Map to grid boundary', default=False, description=desc['map_to_boundary'])

    def _add_flux_nodal_bc(self, gmi_section: Section) -> None:
        """Adds a 'flux nodal' bc.

        Args:
            gmi_section (Section): The group set added to.
        """
        group_name = 'Flux nodal'
        desc = self._desc
        gp = self._add_simple_bc('point_parameters', gmi_section, group_name, [Domains.PM, Domains.OLF])

        name_label = ('nodal_flux_reduction_by_pressure_head', 'Nodal flux reduction by pressure head')  # for short
        gp.add_boolean(name_label[0], name_label[1], default=False, description=desc[name_label[0]])
        parent = ['point_parameters', group_name, 'nodal_flux_reduction_by_pressure_head']

        p = gp.add_float('min_pressure_head', 'Minimum pressure head', 0.0, description=desc['min_pressure_head'])
        p.add_dependency(parent, flags={True: True, False: False})

        p = gp.add_float('max_pressure_head', 'Maximum pressure head', 0.0, description=desc['max_pressure_head'])
        p.add_dependency(parent, flags={True: True, False: False})

    def _add_critical_depth_bc(self, gmi_section: Section) -> None:
        """Adds a Critical depth bc.

        Args:
            gmi_section (Section): The group set added to.
        """
        desc = self._desc
        group_name = 'Critical depth'
        gp = gmi_section.add_group(group_name=group_name, label='Critical depth', description=desc[group_name])

        gp.add_text('name', 'Name', default=_defaultize(group_name), description=desc['name'], required=True)

        gp.add_boolean('map_to_boundary', 'Map to grid boundary', default=False, description=desc['map_to_boundary'])

    def _add_point_bcs(self, gm: GenericModel) -> None:
        """Adds the point BCs."""
        pp = gm.point_parameters
        pp.exclusive_groups = True
        self._add_flux_nodal_bc(pp)

    def _add_arc_bcs(self, gm: GenericModel) -> None:
        """Adds the arc BCs."""
        ap = gm.arc_parameters
        ap.exclusive_groups = True
        self._add_arc_bc(ap, 'Head', domains=[Domains.PM, Domains.OLF])
        self._add_arc_bc(ap, 'Flux', domains=[Domains.PM, Domains.OLF])
        self._add_arc_bc(ap, 'Simple river', domains=[Domains.OLF])
        self._add_arc_bc(ap, 'Simple drain', domains=[Domains.OLF])
        self._add_critical_depth_bc(ap)  # Only OLF domain

    def _add_polygon_bcs(self, gm: GenericModel) -> None:
        """Adds the polygon BCs."""
        pp = gm.polygon_parameters
        pp.exclusive_groups = True
        self._add_simple_bc('polygon_parameters', pp, 'Head', domains=[Domains.PM, Domains.OLF])
        self._add_simple_bc('polygon_parameters', pp, 'Flux', domains=[Domains.PM, Domains.OLF])
        self._add_simple_bc('polygon_parameters', pp, 'Rain', domains=[Domains.OLF])
        self._add_simple_bc('polygon_parameters', pp, 'Potential evapotranspiration', domains=[Domains.ET])


def _descriptions() -> dict[str, str]:
    """Returns a dict of descriptions."""
    return {
        'name':
            'Name of the boundary condition. Names must be unique (case-insensitive comparison) and should not contain'
            ' any whitespace characters.',
        'domain':
            'The domain in which the boundary condition is applied.',
        'range_opts':
            'Choose how the vertical range will be specified',
        'max_sheet':
            'Maximum node sheet. BC will be applied between the minimum and maximum node sheets.',
        'min_sheet':
            'Minimum node sheet. BC will be applied between the minimum and maximum node sheets.',
        'top_elev':
            'Top of the range [L]. BC will be applied between the top and bottom elevations.',
        'bottom_elev':
            'Bottom of the range [L]. BC will be applied between the top and bottom elevations.',
        'input_type':
            'How the boundary condition data is specified.',
        'constant':
            'Constant value of the boundary condition.',
        'constant_start':
            'Constant value of the boundary condition at the start of the arc.',
        'constant_end':
            'Constant value of the boundary condition at the end of the arc.',
        'xy_series':
            'Time-varying values of the boundary condition.',
        'xy_series_start':
            'Time-varying values of the boundary condition at the start of the arc.',
        'xy_series_end':
            'Time-varying values of the boundary condition at the end of the arc.',
        'interpolate':
            'Causes time-varying values to be interpolated, resulting in a smoother application of the '
            'time-varying function.',
        'map_to_boundary':
            'Map BC to the boundary of the grid (not the interior).',
        'nodal_flux_reduction_by_pressure_head':
            'Reduces flux as a function of the nodal pressure head to avoid pumping from the dry materials',
        'min_pressure_head':
            'Minimum pressure head threshold.',
        'max_pressure_head':
            'Maximum pressure head threshold.',
        'transient':
            'Option to define a time value table to turn the BC on and off. Values of -99999 indicate that'
            ' the BC is turned off.',
        'xy_critical_depth':
            'Time value table. Values of -99999 indicate that the BC is turned off.',
        'Flux nodal':
            'Specified flux boundary condition. Treats fluxes as nodal volumetric fluxes [L^3*T^-1] that are applied '
            'directly to nodes.',
        'Head':
            'Specified head boundary condition, applied to nodes.',
        'Flux':
            'Specified flux boundary condition, applied to faces.',
        'Simple river':
            'River flux boundary condition, applied to nodes.',
        'Simple drain':
            'Drain flux boundary condition, applied to nodes',
        'Critical depth':
            'Critical depth boundary condition, assigned to segments which should be part of the overland flow domain '
            'and are typically located on the outer boundary.',
        'Rain':
            'Specified flux boundary condition, applied to faces.',
        'Potential evapotranspiration':
            'Potential evapotranspiration flux [L T^-1].',
    }


def _defaultize(name: str) -> str:
    """Returns the name with spaces replaced by '-' and lower case."""
    return name.lower().replace(' ', '-')
