"""Ah2FileReader class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
import logging
from pathlib import Path
from typing import Any, TextIO

# 2. Third party modules

# 3. Aquaveo modules
from xms.constraint import UGrid2d
from xms.grid.ugrid import UGrid

# 4. Local modules

Location = tuple[float, float, float]
Triangle = tuple[int, int, int]


def read(filepath: Path | str, logger: logging.Logger | None = None) -> UGrid2d:
    """Import an AH2 formatted file, which is almost identical to a GMS TIN file, and return a 2D UGrid.

    Args:
        filepath: Path to .ah2 file.
        logger: Optional logger.

    Returns:
        See description.
    """
    with open(filepath, 'r') as file:
        # Read points
        locations = _read_locations(file)
        _log(f'{len(locations)} locations read.', logger, logging.INFO)

        # Read triangles
        triangles = _read_triangles(file)
        _log(f'{len(triangles)} triangles read.', logger, logging.INFO)

        # Create the cogrid
        cogrid = _create_cogrid(locations, triangles)
        _log('UGrid created.', logger, logging.INFO)

        return cogrid


def _log(message: str, logger: logging.Logger | None, level: int) -> None:
    """Add message to log if logger is not None.

    Args:
        message: The message.
        logger: The logger.
        level: The log level.
    """
    if logger:
        logger.log(level, message)


def _read_locations(file: TextIO) -> list[Location]:
    """Read the number of locations and the locations and return list of xyz tuples.

    Args:
        file: The .ah2 file.

    Returns:
        See description.
    """
    num_xy = int(file.readline().strip())
    locations = []
    for _ in range(num_xy):
        words = file.readline().strip().split()
        locations.append((float(words[0]), float(words[1]), 0.0))
    return locations


def _read_triangles(file: TextIO) -> list[Triangle]:
    """Read the triangles and return a list of tuples, with point numbers decremented by 1 to make them 0-based.

    Args:
        file: The .ah2 file.

    Returns:
        See description.
    """
    num_tri = int(file.readline().strip())
    triangles = []
    for _ in range(num_tri):
        words = file.readline().strip().split()
        triangles.append((int(words[0]) - 1, int(words[1]) - 1, int(words[2]) - 1))
    return triangles


def _make_cell_stream(triangles: list[Triangle]) -> list[Any]:
    """Returns the cell stream.

    Args:
        triangles: list of triangles.

    Returns:
        See description.
    """
    cell_type = UGrid.cell_type_enum.TRIANGLE
    cell_stream = []
    cell_points_count = 3
    for triangle in triangles:
        cell_stream.extend([int(cell_type), cell_points_count, *triangle])
    return cell_stream


def _create_cogrid(locations: list[Location], triangles: list[Triangle]) -> UGrid2d:
    """Create and return the UGrid2d.

    Args:
        locations: List of locations.
        triangles: Lis of triangles.

    Returns:
        See description.
    """
    cell_stream = _make_cell_stream(triangles)
    ugrid = UGrid(locations, cell_stream)
    cogrid = UGrid2d(ugrid)
    return cogrid
