"""HydroAsComponent base class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'
__all__ = ['CoverageComponent']

# 1. Standard Python modules
from functools import cached_property
from pathlib import Path
from typing import Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.bases.component_with_menus_base import MessagesAndRequests
from xms.gmi.component_bases.coverage_component_base import CoverageComponentBase
from xms.gmi.data.generic_model import GenericModel, Section
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.hydroas.components.template_finder import find_template
from xms.hydroas.data.coverage_data import CoverageData


class CoverageComponent(CoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base for the HYDRO_AS-2D model."""
    def __init__(self, main_file: Optional[str | Path] = None):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self._create_event_handlers.append(self._check_template)
        self.point_dlg_title = 'HydroAS Boundary Condition Parameters'
        self.arc_dlg_title = 'HydroAS Boundary Condition Parameters'

    @cached_property
    def data(self) -> CoverageData:
        """The component's data manager."""
        return CoverageData(self.main_file)

    def _get_section(self, target: TargetType) -> Section:
        """Get a Section with parameters for a given target."""
        return self.data.generic_model.section_from_target_type(target)

    def _check_template(self, _query: Query):
        """Check if the component needs its template initialized and, if necessary, schedule the initialization."""
        if self.data.generic_model.is_default():
            action = self._make_request(self._get_template)
            self.requests.append(action)

    def _get_template(self, query: Query, _params: list[dict], parent: QWidget) -> MessagesAndRequests:
        """
        Initialize the component's template.

        Args:
            query: Interprocess communication object.
            _params: Ignored.
            parent: Parent widget for dialogs.

        Returns:
            Messages and requests.
        """
        find_template(self.data, query, parent)
        # The base class does this too, but it happens before we initialize the template. Without a template, the
        # feature list will be empty, so the display .json files won't be created. Now that we have a template, we'll
        # initialize the .json files again.
        self._initialize_display_json_files()
        return [], []

    def to_text_file(self, model: GenericModel, file: str | Path):
        """
        Export the component to a text file suitable for comparing to a test baseline.

        This is intended to be used after building the component and preparing it to be sent to XMS. The output data
        includes all the attributes assigned in self.data, the UUID of the coverage the component expects to be attached
        to, and a mapping of feature and component IDs to verify that the right things were associated with each other.

        Args:
            model: Ignored. Only present for compatibility with base class.
            file: Where to export to.
        """
        super().to_text_file(self.data.generic_model, file)
