"""Module for the RoughnessComponent."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['RoughnessComponent', 'get_section']

# 1. Standard Python modules
from pathlib import Path
from typing import Optional

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.component_bases.coverage_component_base import CoverageComponentBase
from xms.gmi.data.generic_model import GenericModel, Section
from xms.guipy.data.target_type import TargetType

# 4. Local modules


class RoughnessComponent(CoverageComponentBase):
    """A Dynamic Model Interface (DMI) component base for the HYDRO_AS-2D model."""
    def __init__(self, main_file: Optional[str | Path]):
        """Initialize the component."""
        super().__init__(main_file)
        self._section_dialog_keyword_args |= {'enable_unchecked_groups': True, 'show_groups': False}
        self.polygon_dlg_title = 'HydroAS Bridge Properties'

    def _get_section(self, target: TargetType) -> Section:
        """Get a Section with parameters for a given target."""
        if target == TargetType.polygon:
            return get_section()
        else:
            # No special reason for model_parameters. Just need an empty section so menus don't appear for this target.
            return GenericModel().model_parameters


def get_section() -> Section:
    """Get the GenericModel section for the roughness component."""
    model = GenericModel()
    group = model.material_parameters.add_group('bridge', 'Bridge Properties', is_active=True)
    group.add_float('roughness', 'Roughness (K_St in m^(1/3)/s)', 6.66666667e+01, low=1.0)
    group.add_dataset('elevation', 'Elevation [abs. Height]')
    return model.material_parameters
