"""Module for ImportSimRunner class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['ImportSimRunner']

# 1. Standard Python modules
from typing import cast

# 2. Third party modules

# 3. Aquaveo modules
from xms.components.component_builders.coverage_component_builder import CoverageComponentBuilder
from xms.components.component_builders.main_file_maker import create_component, make_main_file
from xms.guipy.data.target_type import TargetType
from xms.guipy.dialogs.feedback_thread import ExpectedError, FeedbackThread

# 4. Local modules
from xms.hydroas.components.coverage_component import CoverageComponent
from xms.hydroas.components.mapped_material_component import MappedMaterialComponent
from xms.hydroas.components.material_component import MaterialComponent
from xms.hydroas.components.sim_component import SimComponent
from xms.hydroas.data.coverage_data import CoverageData
from xms.hydroas.data.data import get_version
from xms.hydroas.data.material_data import MaterialData
from xms.hydroas.dmi.xms_data import XmsData
from xms.hydroas.file_io.errors import GmiError
from xms.hydroas.file_io.gmi_reader import GmiReader


class ImportSimRunner(FeedbackThread):
    """Import thread."""
    def __init__(self, data: XmsData):
        """
        Construct the worker.

        Args:
            data: Interprocess communication object.
        """
        super().__init__(create_query=False)
        self.display_text |= {
            'title': 'Reading .2dm Geometry File',
            'working_prompt': 'Reading .2dm geometry file. Please wait...',
            'error_prompt': 'Error(s) encountered reading .2dm file. Review log output for more details.',
            'warning_prompt': 'Warning(s) encountered reading .2dm file. Review log output for more details.',
            'success_prompt': 'Successfully read .2dm geometry file.',
        }
        self._data = data
        self._sim_uuid = ''

    def _run(self):
        """Run the thread."""
        self._log.info('Importing Hydro-AS simulation...')
        reader = GmiReader(self._data.import_file, log=self._log)
        try:
            reader.read()
        except GmiError as exc:
            raise ExpectedError(str(exc))
        reader.model.arc_parameters.exclusive_groups = True

        self._version = get_version(reader.model, reader.model_instantiation)

        self._add_simulation(reader)

        if reader.ugrid is not None:
            self._data.add_ugrid(reader.ugrid, reader.mesh_name, reader.projection)

        self._add_point_coverage(reader)
        self._add_arc_coverage(reader)
        self._add_material_coverage(reader)
        self._add_mapped_materials(reader)
        self._data.finish()

    def _add_simulation(self, reader: GmiReader):
        """
        Add the simulation.

        Args:
            reader: Reader with simulation data to add.
        """
        self._log.info('Creating simulation...')
        sim = create_component(SimComponent)
        sim.data.global_values = reader.global_instantiation
        sim.data.model_values = reader.model_instantiation
        sim.data.generic_model = reader.model
        sim.data.commit()

        name = f'{reader.mesh_name} [{self._version}]'

        self._sim_uuid = sim.uuid
        self._data.add_sim(name, sim)

    def _add_point_coverage(self, reader: GmiReader):
        """
        Add the point coverage.

        Args:
            reader: Reader with points to add.
        """
        if not reader.points:
            return

        self._log.info('Creating point coverage...')
        name = f'HydroAS Point BCs [{self._version}]'

        builder = CoverageComponentBuilder(CoverageComponent, name, reader.ugrid)
        data: CoverageData = cast(CoverageData, builder.data)  # TODO: Make CoverageData inherit to enforce this
        component_ids = data.add_features(TargetType.point, reader.point_values, reader.point_types)
        data.generic_model = reader.model

        for point, component_id in zip(reader.points, component_ids, strict=True):
            builder.add_node(point, component_id)

        builder.turn_off_labels()
        coverage, component, keywords = builder.build()
        coverage.projection = reader.projection

        self._data.add_coverage(coverage, component, keywords=keywords)
        self._data.link(coverage.uuid)

    def _add_arc_coverage(self, reader: GmiReader):
        """
        Add the arc coverage.

        Args:
            reader: Reader with arcs to add.
        """
        if not reader.arcs:
            return

        self._log.info('Creating arc coverage...')
        name = f'HydroAS Arc BCs [{self._version}]'

        builder = CoverageComponentBuilder(CoverageComponent, name, reader.ugrid)
        data: CoverageData = cast(CoverageData, builder.data)  # TODO: Make CoverageData inherit to enforce this
        component_ids = data.add_features(TargetType.arc, reader.arc_values, reader.arc_types)
        data.generic_model = reader.model

        for arc, component_id in zip(reader.arcs, component_ids, strict=True):
            builder.add_node_string(arc, component_id)
        builder.turn_off_labels()
        coverage, component, keywords = builder.build()
        coverage.projection = reader.projection

        self._data.add_coverage(coverage, component, keywords=keywords)
        self._data.link(coverage.uuid)

    def _add_material_coverage(self, reader: GmiReader):
        """
        Add the material coverage.

        Args:
            reader: Reader with materials to add.
        """
        if len(reader.material_cells) == 0:
            return

        self._log.info('Building material dataset...')

        name = f'HydroAS Materials [{self._version}]'

        builder = CoverageComponentBuilder(MaterialComponent, name, reader.ugrid)
        data: CoverageData = cast(MaterialData, builder.data)  # TODO: Make MaterialData inherit to enforce this
        data.material_values = reader.material_instantiation
        data.generic_model = reader.model
        section = reader.model.copy().material_parameters

        values = []
        for material_type in reader.material_types:
            section.clear_values()
            section.group(material_type).is_active = True
            values.append(section.extract_group_activity())
        component_ids = data.add_features(TargetType.polygon, values, reader.material_types)
        dataset_value_to_component_id = {
            int(material_type): component_id
            for material_type, component_id in zip(reader.material_types, component_ids)
        }

        self._log.info('Building material polygons...')
        builder.add_polygons(reader.material_cells, dataset_value_to_component_id, null_value=0)
        self._log.info('Building material coverage...')
        coverage, component, keywords = builder.build()
        coverage.projection = reader.projection

        self._data.add_coverage(coverage, component, keywords=keywords)

    def _add_mapped_materials(self, reader: GmiReader):
        """Add mapped materials to XMS."""
        main_file = make_main_file(MappedMaterialComponent)
        component = MappedMaterialComponent(main_file)
        data = component.data
        data.generic_model = reader.model
        data.material_values = reader.material_instantiation
        data.cell_materials = reader.material_cells
        if reader.ugrid:
            data.grid_hash = self._data.grid_hash
        data.commit()
        self._data.add_mapped_materials(component)
        self._data.link(component.uuid)
