"""Functions to help with duplicating a simulation."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path
import re

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as fs
from xms.guipy import file_io_util

# 4. Local modules
from xms.mf6.file_io import io_util

# Constants used when duplicating a simulation
DUPLICATION_HELPER_FILE_NAME = 'mf6.txt'  # Name of the file used to help when duplicating a simulation
LAST_ITEM_UUID = 'last_item_uuid'  # Constant string to identify the last item being duplicated
OLD_NEW_MAIN_FILES = 'old_new_main_files'  # Main files of items to be duplicated


def _multireplace(string: str, replacements: dict[str, str], ignore_case: bool = False) -> str:
    """Given a string and a replacement map, it returns the replaced string.

    https://stackoverflow.com/questions/6116978/how-to-replace-multiple-substrings-of-a-string
    https://gist.github.com/bgusach/a967e0587d6e01e889fd1d776c5f3729

    Args:
        string: string to execute replacements on
        replacements: replacement dictionary {value to find: value to replace}
        ignore_case: whether the match should be case insensitive

    Returns:
        (str): The fixed string.
    """
    if not replacements:
        # Edge case that'd produce a funny regex and cause a KeyError
        return string

    # If case insensitive, we need to normalize the old string so that later a replacement
    # can be found. For instance with {"HEY": "lol"} we should match and find a replacement for "hey",
    # "HEY", "hEy", etc.
    if ignore_case:

        def normalize_old(s):
            return s.lower()

        re_mode = re.IGNORECASE

    else:

        def normalize_old(s):
            return s

        re_mode = 0

    replacements = {normalize_old(key): val for key, val in replacements.items()}

    # Place longer ones first to keep shorter substrings from matching where the longer ones should take place
    # For instance given the replacements {'ab': 'AB', 'abc': 'ABC'} against the string 'hey abc', it should produce
    # 'hey ABC' and not 'hey ABc'
    rep_sorted = sorted(replacements, key=len, reverse=True)
    rep_escaped = map(re.escape, rep_sorted)

    # Create a big OR regex that matches any of the substrings to replace
    pattern = re.compile("|".join(rep_escaped), re_mode)

    # For each match, look up the new string in the replacements, being the key the normalized old string
    return pattern.sub(lambda match: replacements[normalize_old(match.group(0))], string)


def _replace_uuids_in_file(filepath: str, old_to_new_uuids: dict[str, str]) -> None:
    """Replace old uuids in file with new ones.

    Args:
        filepath (str): Path to file.
        old_to_new_uuids (dict[str, str]): Dict of old uuids and their new uuids.
    """
    # Read the file as one string
    file_string = ''  # The entire file as one string
    with open(filepath, 'r') as file:
        file_string = file.read()

    # Overwrite the original file with the new lines
    new_string = _multireplace(file_string, old_to_new_uuids)
    if Path(filepath).is_file():  # Needed to fix weird test failures (
        with open(filepath, 'w') as file:
            file.write(new_string)


def _replace_all_uuids_in_folder(new_main_file: str, old_to_new_uuids: dict[str, str]) -> None:
    """Replace any old uuids in all files in the new component folder, recursively.

    Args:
        new_main_file (str): Path to the new main file.
        old_to_new_uuids (dict[str, str]): Dict of old uuids and their new uuids.
    """
    filepaths = Path(new_main_file).parent.glob('**/*')
    for filepath in filepaths:
        if filepath.is_file() and not io_util.file_is_binary(filepath) and filepath.suffix not in {'.grb'}:
            _replace_uuids_in_file(str(filepath), old_to_new_uuids)


def _replace_old_uuids(data: dict) -> None:
    """Replace all old uuids with new ones in all duplicated folders.

    Args:
        data (dict): Dict with data from duplication helper file.
    """
    # Get a dict of all the old uuids and their new uuids
    files = data[OLD_NEW_MAIN_FILES]
    old_to_new_uuids = {io_util.uuid_from_path(o): io_util.uuid_from_path(n) for o, n in files.items()}

    for _old_main_file, new_main_file in data[OLD_NEW_MAIN_FILES].items():
        _replace_all_uuids_in_folder(new_main_file, old_to_new_uuids)


def get_helper_filepath() -> Path:
    """Returns the path to the helper file used while the simulation is being duplicated.

    Returns:
        (Path): See description.
    """
    return Path(io_util.get_xms_temp_directory()) / DUPLICATION_HELPER_FILE_NAME


def add_file_to_duplication_helper_file(old_main_file: Path | str, new_main_file: Path | str) -> dict:
    """Adds the file to the stack in the duplicating helper file.

    The duplication helper file is a temporary file used while a sim is being duplicated that helps us fix the
    paths in the new main files to use the new uuids of the new components. It is json and includes a list
    of old and new main files, and the last file that will be duplicated. All the files in the component folders
    will be fixed so that if they referred to other old files, they will refer to the new files.

    Args:
        old_main_file: Path to the old main file.
        new_main_file: Path to the new main file.

    Returns:
        (dict): Dict of the file contents.
    """
    old_main_file = str(old_main_file)
    new_main_file = str(new_main_file)
    helper_filepath = get_helper_filepath()
    data = file_io_util.read_json_file(helper_filepath)
    if OLD_NEW_MAIN_FILES not in data:
        data[OLD_NEW_MAIN_FILES] = {old_main_file: new_main_file}
    elif old_main_file not in data[OLD_NEW_MAIN_FILES]:
        data[OLD_NEW_MAIN_FILES][old_main_file] = new_main_file
    file_io_util.write_json_file(data, helper_filepath)
    return data


def handle_if_last_duplicated_item(old_main_file: Path | str, data: dict, single_component: bool) -> None:
    """Check if this is the last duplicated item, and if so, replace all old uuids and remove helper file.

    Args:
        old_main_file: Path to old main file.
        data: Dict with data from duplication helper file.
        single_component: True if we are only duplicating a single component.
    """
    old_uuid = io_util.uuid_from_path(old_main_file)
    last_item_uuid = data.get(LAST_ITEM_UUID)
    if single_component or (last_item_uuid and old_uuid == last_item_uuid):
        _replace_old_uuids(data)
        fs.removefile(str(get_helper_filepath()))
